﻿# tweetutilities.py

"""Funkcje pomocnicze do interakcji z obiektami biblioteki Tweepy."""
from geopy import OpenMapQuest
import keys
from textblob import TextBlob 
import time 
import tweepy

def get_API(wait=True, notify=True):
    """Uwierzytelnianie w Twitterze i utworzenie obiektu api."""
    # konfigurowanie obiektu OAuthHandler
    auth = tweepy.OAuthHandler(keys.consumer_key, keys.consumer_secret)
    auth.set_access_token(keys.access_token, keys.access_token_secret)

    # utworzenie i zwrócenie obiektu api
    return tweepy.API(auth, wait_on_rate_limit=wait, 
                      wait_on_rate_limit_notify=notify)

def print_tweets(tweets):
    """Dla każdego obiektu Status w kolekcji tweets wypisywana
    jest nazwa użytkownika (screen_name) i treść tweetu (text). 
    Ponadto, jeżeli język tweetu nie jest określony jako 
    niezdefiniowany (und), wypisywany jest wynik tłumaczenia
    treści na język angielski."""
    
    for tweet in tweets:
        print(f'{tweet.user.screen_name}:', end=' ')
    
        if 'en' in tweet.lang:
            print(f'{tweet.text}\n')
        elif 'und' not in tweet.lang:  # tłumaczenie na język angielski
            print(f'\n   ORYGINAŁ: {tweet.text}')
            print(f'TŁUMACZENIE: {TextBlob(tweet.text).translate()}\n')

def get_tweet_content(tweet, location=False):
    """Zwraca słownik z danyumi pobranymi z obiektu Status."""
    fields = {}
    fields['screen_name'] = tweet.user.screen_name

    # pobierz treść tweetu
    try:  
        fields['text'] = tweet.extended_tweet.full_text
    except: 
        fields['text'] = tweet.text

    if location:
        fields['location'] = tweet.user.location

    return fields

def get_geocodes(tweet_list):
    """Pobieranie informacji o lokalizacji (szerokości i długości
    geograficznej) nadawcy każdego tweetu z listy. Jako wynik 
    zwracana jest liczba tweetów z błędnymi danymi lokalizacyjnymi"""     

    print('Pobieranie informacji o lokalizacji tweetów...')
    geo = OpenMapQuest(api_key=keys.mapquest_key)  # geocoder
    bad_locations = 0  

    for tweet in tweet_list:
        processed = False
        delay = .1  # wykorzystywany w przypadku przeterminowania
                    # wywołania OpenMapQuest, do opóźnienia powtórnego
                    # wywołania
    
        while not processed:
            try:  # pobieranie współrzędnych lokalizacji tweetu
                  # z tweet['location']
                geo_location = geo.geocode(tweet['location'])
                processed = True
            except:  # przeterminowanie, opóźnij powtórne wywołanie
                print('OpenMapQuest: upłynął limit czasu oczekiwania. Ponawiam próbę')
                time.sleep(delay)
                delay += .1

        if geo_location:  
            tweet['latitude'] = geo_location.latitude
            tweet['longitude'] = geo_location.longitude
        else:  
            bad_locations += 1  # tweet['location'] zawiera błędne dane
    
    print('Gotowe.')
    return bad_locations


##########################################################################
# (C) Copyright 2019 by Deitel & Associates, Inc. and                    #
# Pearson Education, Inc. All Rights Reserved.                           #
#                                                                        #
# DISCLAIMER: The authors and publisher of this book have used their     #
# best efforts in preparing the book. These efforts include the          #
# development, research, and testing of the theories and programs        #
# to determine their effectiveness. The authors and publisher make       #
# no warranty of any kind, expressed or implied, with regard to these    #
# programs or to the documentation contained in these books. The authors #
# and publisher shall not be liable in any event for incidental or       #
# consequential damages in connection with, or arising out of, the       #
# furnishing, performance, or use of these programs.                     #
##########################################################################
