# SimpleLanguageTranslator.py
#-*- coding: 1250 -*-
   
""" Aplikacja uatwiajca porozumiewanie si dwch rozmwcw,
    z ktrych jeden posuguje si jzykiem angielskim, a drugi 
    hiszpaskim. Wykorzystuje API nastpujcych usug platformy
    IBM Watson:
       
       Speech to Text
       Text to Speech
       Language Translator
"""   
   
from watson_developer_cloud import SpeechToTextV1
from watson_developer_cloud import LanguageTranslatorV3
from watson_developer_cloud import TextToSpeechV1
import keys  # zawiera klucze API dla poszczeglnych usug
import pyaudio  # wykorzystywany do nagrywania przez mikrofon
import pydub  # wykorzystywany do adowania pliku WAV
import pydub.playback  # wykorzystywany do odtwarzania pliku WAV
import wave  # wykorzystywany do zapisywania w pliku WAV

def prepare_input(aMsg1, aMsg2, aFilenameRecord, aFileNameDefault):
    print()
    print('Wprowad:')
    print('  n - by nagra ' + aMsg1)
    print('  g - by wykorzysta ' + aMsg2 + ' (' + aFileNameDefault + ')')
    print('i nacinij Enter')
    print()
    ready = False
    while not ready:
      k = input(' ?')
      
      ready = (k == 'n') or (k == 'g')
      
      if not ready:
          print('Nieprawidowy klawisz - wprowad ponownie')
    
    if k == 'n':   # nagrywanie
        print('Zacznij mwi - masz 5 sekund')
        record_audio(aFilenameRecord)
        play_audio(aFilenameRecord)
        result_filename = aFilenameRecord
    else:
        print('Wykorzystuj zawarto pliku ' + aFileNameDefault) 
        result_filename = aFileNameDefault    
        play_audio(file_name=aFileNameDefault)
        
    return result_filename        
        
def run_translator():
    """Wywouje funkcje realizujce interakcj 
       z usugami IBM Watson"""
   
    # Krok 1. Sygna do zadania pytania w jzyku angielskim
    #         i nagrywanie gosu do pliku audio
    english_question_file_name = prepare_input(
                                         'pytanie po angielsku',
                                         'nagrane pytanie',
                                         'english_prepared.wav',
                                         'english.wav'
                                              )

    # Krok 2. Konwersja mowy angielskiej na tekst 
    english = speech_to_text(
        file_name=english_question_file_name, model_id='en-US_BroadbandModel')
    print('Angielski: ', english)

    # Krok 3. Tumaczenie tekstu angielskiego na jzyk hiszpaski
    spanish = translate(text_to_translate=english, model='en-es')
    print('Hiszpaski: ', spanish)

    # Krok 4. Synteza gosu na podstawie tekstu hiszpaskiego
    text_to_speech(text_to_speak=spanish, voice_to_use='es-US_SofiaVoice',
        file_name='spanish.wav')

    # Krok 5. Odtworzenie pliku audio ze zsyntetyzowan 
    # mow hiszpask
    play_audio(file_name='spanish.wav')

    # Krok 6. Sygna do udzielenia odpowiedzi w jzyku hiszpaskim
    #         i nagrywanie gosu do pliku audio
    spanish_response_file_name = prepare_input(
                                         'odpowied po hiszpasku',
                                         'nagran odpowied',
                                         'spanishresponse_prepared.wav',
                                         'spanishresponse.wav'
                                              )
    
    # Krok 7. Konwersja mowy hiszpaskiej na tekst
    spanish = speech_to_text(
        file_name=spanish_response_file_name, model_id='es-ES_BroadbandModel')
    print('Odpowied po hiszpasku: ', spanish)

    # Krok 8. Tumaczenie tekstu hiszpaskiego na jzyk angielski
    english = translate(text_to_translate=spanish, model='es-en')
    print('Odpowied po angielsku: ', english)

    # Krok 9. Synteza gosu na podstawie tekstu angielskiego
    text_to_speech(text_to_speak=english,
        voice_to_use='en-US_AllisonVoice',
        file_name='englishresponse.wav')

    # Krok 10. Odtworzenie pliku audio ze zsyntetyzowan 
    # mow angielsk
    play_audio(file_name='englishresponse.wav')


def speech_to_text(file_name, model_id):
    """Wywoanie usugi Watson Speech to Text w celu
       skonwertowania gosu na tekst pisany""" 
    
    # utworzenie klienta usugi
    stt = SpeechToTextV1(iam_apikey=keys.speech_to_text_key)

    # otwarcie pliku audio
    with open(file_name, 'rb') as audio_file:
        # przekazanie zawartoci pliku do konwersji
        result = stt.recognize(audio=audio_file,
            content_type='audio/wav', model=model_id).get_result()
    
    # Pobranie listy wynikw. Moe ona zawiera, oprcz wyniku
    # kocowego, take wyniki porednie, zalenie od danej metody
    # rozpoznawania gosu. W tym przypadku damy tylko kocowego
    # wyniku, wic lista zawiera tylko jeden element. 
    
    results_list = result['results'] 

    # Pobierz kocowy wynik rozpoznawania
    speech_recognition_result  = results_list[0]
    
    # Pobranie listy wariantw transkrypcji.Jej liczebno
    # zalena jest od danej metody rozpoznawania gosu.
    # W tym przypadku lista zawiera tylko jeden element.
    alternatives_list = speech_recognition_result['alternatives']
    
    first_alternative = alternatives_list[0]

    # Pobranie tekstu stanowicego wynik transkrypcji
    # i zwrcenie go jako wynik
    transcript = first_alternative['transcript']
    return transcript


def translate(text_to_translate, model):
    """Wywoanie usugi Watson Language Translator w celu
       przetumaczenia tekstu angielskiego na hiszpaski
       (model en-es) albo hiszpaskiego na angielski 
       (model es-en)"""   
   
    # utworzenie klienta usugi
    language_translator = LanguageTranslatorV3(version='2018-05-31',
        iam_apikey=keys.translate_key)

    # wykonanie tumaczenia
    translated_text = language_translator.translate(
        text=text_to_translate, model_id=model).get_result()
    
    # Pobranie listy wynikowych acuchw - kady z nich 
    # odpowiada jednemu acuchowi rdowemu. W tym przypadku
    # przekazujemy tylko jeden acuch rdowy, wic lista 
    # wynikw zawiera tylko jeden element.
    translations_list = translated_text['translations']
    first_translation = translations_list[0]

    # pobranie przetumaczonego tekstu i zwrcenie go 
    # jako wynik
    translation = first_translation['translation']
    return translation

def text_to_speech(text_to_speak, voice_to_use, file_name):
    """Wywoanie usugi Watson Text to Speech w celu
       zsyntetyzowania mowy, z dan barw gosu i zapisanie
       wyniku syntezy w pliku WAV o danej nazwie."""
       
    # utworzenie klienta usugi
    tts = TextToSpeechV1(iam_apikey=keys.text_to_speech_key)

    # utworzenie pliku, do ktrego zapisany zostanie 
    # wynik syntezy
    with open(file_name, 'wb') as audio_file:
        audio_file.write(tts.synthesize(text_to_speak, 
            accept='audio/wav', voice=voice_to_use).get_result().content)

def record_audio(file_name):
    """Zapisanie krtkiego (do 5 sekund) nagrania
       w pliku WAV o podanej nazwie, z wykorzystaniem 
       moduu pyaudio"""
    
    FRAME_RATE = 44100  # liczba ramek na sekund
    CHUNK = 1024  # liczba ramek odczytywanych jednorazowo
    FORMAT = pyaudio.paInt16  # kada ramka jest 16-bitow 
                              # liczb cakowit
    CHANNELS = 2  # 2 sample na ramk
    SECONDS = 5  # cakowity czas nagrania
 
    # otwarcie i zamknicie strumieni audio
    recorder = pyaudio.PyAudio() 

    # skonfigurowanie i otwarcie strumienia audio
    # do nagrywania (input=True)
    audio_stream = recorder.open(format=FORMAT, channels=CHANNELS, 
        rate=FRAME_RATE, input=True, frames_per_buffer=CHUNK)
   
    # Zapisywanie surowych bajtw z wejcia mikrofonowego
    audio_frames = []  
    # print('Nagrywanie 5-sekundowe.')

    # odczytywanie nagrania w porcjach okrelonych przez CHUNK
    for i in range(0, int(FRAME_RATE * SECONDS / CHUNK)):
        audio_frames.append(audio_stream.read(CHUNK))

    print('Nagrywanie zakoczone.')
    audio_stream.stop_stream()  # zatrzymanie nagrywania
    audio_stream.close()  
    recorder.terminate()  # zwolnienie zasobw uywanych przez pyaudio

    # zapisanie ramek audio w pliku WAV
    with wave.open(file_name, 'wb') as output_file:
        output_file.setnchannels(CHANNELS)
        output_file.setsampwidth(recorder.get_sample_size(FORMAT))
        output_file.setframerate(FRAME_RATE)
        output_file.writeframes(b''.join(audio_frames))

def play_audio(file_name):
    """Odtworzenie zawartoci pliku WAV o podanej nazwie
       za pomoc moduu pydub"""
    sound = pydub.AudioSegment.from_wav(file_name)
    pydub.playback.play(sound)

if __name__ == '__main__':
    run_translator()


##########################################################################
# (C) Copyright 2019 by Deitel & Associates, Inc. and                    #
# Pearson Education, Inc. All Rights Reserved.                           #
#                                                                        #
# DISCLAIMER: The authors and publisher of this book have used their     #
# best efforts in preparing the book. These efforts include the          #
# development, research, and testing of the theories and programs        #
# to determine their effectiveness. The authors and publisher make       #
# no warranty of any kind, expressed or implied, with regard to these    #
# programs or to the documentation contained in these books. The authors #
# and publisher shall not be liable in any event for incidental or       #
# consequential damages in connection with, or arising out of, the       #
# furnishing, performance, or use of these programs.                     #
##########################################################################
