﻿# simulator.py


# (C) Copyright 2019 by Deitel & Associates, Inc. and                    #
# Pearson Education, Inc. All Rights Reserved.                           #
# Revised by A.Grażyński, (C) Copyright 2020 by Helion Publishing


"""Symulator termostatu przyłączonego do internetu,
   publikującego komunikaty JSON dla usługi dweet.io"""
   

import dweepy
import sys
import time
import random

MIN_CELSIUS_TEMP = -25  
MAX_CELSIUS_TEMP = 45 
MAX_TEMP_CHANGE = 2

MIN_SAFE_TEMP = 3
MAX_SAFE_TEMP = 35

# Gdy True, identyfikator dweetów generowany przez usługę
# Opcja GEN_RANDOM_THING_NAME jest bez znaczenia
AUTO_GEN_THING_NAME = True

# Gdy True, identyfikator dweetów jest generowany 
# z udziałem UUID. Gdy False, ustalany jest domyślny
# idenyfikator
GEN_RANDOM_THING_NAME = True

# Gdy True, automatycznie uruchamiany jest podgląd dweetów
AUTO_BROWSE_WWW = True

# argumenty wywołania:
#   1. Liczba komunikatów
#   2. Interwał czasowy między kolejnymi komunikatami 
NUMBER_OF_MESSAGES = int(sys.argv[1]) 
MESSAGE_DELAY = int(sys.argv[2])

if AUTO_GEN_THING_NAME:
    dweeter = ''
elif GEN_RANDOM_THING_NAME:
    import uuid
    dweeter = 'PFPr16-iot-therm-' + str(uuid.uuid4())
    print('Unikalny identyfikator:\n' + dweeter)
    print()
else:
    dweeter = 'temperature-simulator-deitel-python' # unikalna nazwa publikatora

thermostat = {'Location': 'Boston, MA, USA',
              'Temperature': 20, 
              'LowTempWarning': False,
              'HighTempWarning': False}

print('Start symulatora termostatu')

for message in range(NUMBER_OF_MESSAGES):
    # generowanie losowej wartości temperatury, z zachowaniem
    # dopuszczalnej różnicy względem poprzedniej wartości
    # (bo w rzeczywistości temperatura nie zmienia się zbyt gwałtownie)
    thermostat['Temperature'] += random.randrange(
        -MAX_TEMP_CHANGE, MAX_TEMP_CHANGE + 1)
    
    # adjustacja wartości temperatury do dozwolonych granic
    if thermostat['Temperature'] < MIN_CELSIUS_TEMP:
        thermostat['Temperature'] = MIN_CELSIUS_TEMP
    
    if thermostat['Temperature'] > MAX_CELSIUS_TEMP:
        thermostat['Temperature'] = MAX_CELSIUS_TEMP
    
    # kontrola temperatury względem dolnej granicy bezpieczeństwa
    if thermostat['Temperature'] < MIN_SAFE_TEMP:
        thermostat['LowTempWarning'] = True
    else:
        thermostat['LowTempWarning'] = False

    # kontrola temperatury względem górnej granicy bezpieczeństwa
    if thermostat['Temperature'] > MAX_SAFE_TEMP:
        thermostat['HighTempWarning'] = True
    else:
        thermostat['HighTempWarning'] = False
        
    # wysłanie dweetu do usługi dweet.io poprzez bibliotekę Dweepy
    print(f'Wysłano komunikat nr: {message + 1}\r', end='')
    if dweeter == '':
        response = dweepy.dweet(thermostat)
        dweeter = response['thing']
        print('Automatyczny identyfikator:', dweeter)
    else:    
        dweepy.dweet_for(dweeter, thermostat)

    time.sleep(MESSAGE_DELAY)

print('Zatrzymanie symulatora termostatu')

if AUTO_BROWSE_WWW:
    print('Otwieranie przeglądarki:')
    print()
    import webbrowser
    webbrowser.open('https://dweet.io/get/dweets/for/' + dweeter)
