'''
Ten plik wczytuje plik dziennika i generuje plik CSV.
'''
import csv
import argparse
from decimal import Decimal, getcontext

# Ustawianie precyzji na dwa miejsca po przecinku.
getcontext().prec = 2

US_LOCATIONS = ['AL', 'AK', 'AZ', 'AR', 'CA', 'CO', 'CT', 'DE', 'FL', 'GA',
                'HI', 'ID', 'IL', 'IN', 'IA', 'KS', 'KY', 'LA', 'ME', 'MD',
                'MA', 'MI', 'MN', 'MS', 'MO', 'MT', 'NE', 'NV', 'NH', 'NJ',
                'NM', 'NY', 'NC', 'ND', 'OH', 'OK', 'OR', 'PA', 'RI', 'SC',
                'SD', 'TN', 'TX', 'UT', 'VT', 'VA', 'WA', 'WV', 'WI', 'WY',
                'DC']
CAD_LOCATIONS = ['AB', 'BC', 'MB', 'NB', 'NL', 'NS', 'ON', 'PE', 'QC', 'SK',
                 'NT', 'NU', 'YT']
CAD_TO_USD = Decimal(0.76)


def add_price_by_location(row):
    location = row['LOKALIZACJA']
    if location in US_LOCATIONS:
        row['KRAJ'] = 'USA'
        row['WALUTA'] = 'USD'
        row['USD'] = Decimal(row['CENA'])
    elif location in CAD_LOCATIONS:
        row['KRAJ'] = 'KANADA'
        row['WALUTA'] = 'CAD'
        row['USD'] = Decimal(row['CENA']) * CAD_TO_USD
    else:
        raise Exception('Lokalizacji nie znaleziono')

    return row


def main(input_file, output_file):
    reader = csv.DictReader(input_file)
    result = [add_price_by_location(row) for row in reader]

    # Zapisywanie danych w pliku csv.
    header = result[0].keys()
    writer = csv.DictWriter(output_file, fieldnames=header)
    writer.writeheader()
    writer.writerows(result)


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument(dest='input', type=argparse.FileType('r'),
                        help='Plik wejściowy')
    parser.add_argument(dest='output', type=argparse.FileType('w'),
                        help='Plik wyjściowy')
    args = parser.parse_args()
    main(args.input, args.output)
