//: Playground - noun: a place where people can play

import Cocoa

enum Token {
    case Number(Int)
    case Plus
}

class Lexer {
    enum theError: Error {
        case NieprawidłowyZnak(Character)
    }

    let input: String.CharacterView
    var position: String.CharacterView.Index
    
    init(input: String) {
        self.input = input.characters
        self.position = self.input.startIndex
    }
    
    func peek() -> Character? {
        guard position < input.endIndex else {
            return nil
        }
        return input[position]
    }
    
    func advance() {
        assert(position < input.endIndex, "Nie można wykroczyć poza dane wejściowe!")
        position = input.index(after: position)
    }
    
    func getNumber() -> Int {
        var value = 0
        
        while let nextCharacter = peek() {
            switch nextCharacter {
            case "0" ... "9":
                // Kolejna cyfra, dodajemy ją do wartości.
                let digitValue = Int(String(nextCharacter))!
                value = 10*value + digitValue
                advance()
                
            default:
                // To nie jest cyfra, powracamy do analizy leksykalnej.
                return value
            }
        }
        return value
    }
    
    func lex() throws -> [Token] {
        var tokens = [Token]()
        
        while let nextCharacter = peek() {
            switch nextCharacter {
            case "0" ... "9":
                // Rozpoczynamy od cyfry, konieczne jest pobranie reszty.
                let value = getNumber()
                tokens.append(.Number(value))
            case "+":
                tokens.append(.Plus)
                advance()
                
            case " ":
                // Po prostu przechodzimy dalej, aby zignorować spacje.
                advance()
                
            default:
                // Coś nieoczekiwanego, trzeba zgłosić błąd.
                throw theError.NieprawidłowyZnak(nextCharacter)
            }
        }
        return tokens
    }
}

class Parser {
    enum theError: Error {
        case NieoczekiwanyKoniecDanychWejściowych
        case NieprawidłowyToken(Token)
    }

    let tokens: [Token]
    var position = 0
    
    init(tokens: [Token]) {
        self.tokens = tokens
    }
    
    func getNextToken() -> Token? {
        guard position < tokens.count else {
            return nil
        }
        let token = tokens[position]
        position += 1
        return token
    }
    
    func getNumber() throws -> Int {
        guard let token = getNextToken() else {
            throw theError.NieoczekiwanyKoniecDanychWejściowych
        }
        
        switch token {
        case .Number(let value):
            return value
        case .Plus:
            throw theError.NieprawidłowyToken(token)
        }
    }
    
    func parse() throws -> Int {
        // Liczba musi być jako pierwsza.
        var value = try getNumber()
        
        while let token = getNextToken() {
            switch token {
                
            // Dozwolone jest otrzymanie tokenu Plus po Number.
            case .Plus:
                // Po znaku plus konieczne jest otrzymanie kolejnej liczby.
                let nextNumber = try getNumber()
                value += nextNumber
                
            // Niedozwolone jest otrzymanie tokenu Number po Number.
            case .Number:
                throw theError.NieprawidłowyToken(token)
            }
        }
        return value
    }
}

func evaluate(input: String) {
    print("Obliczanie: \(input).")
    let lexer = Lexer(input: input)
    
    do {
        let tokens = try lexer.lex()
        print("Dane wyjściowe analizatora leksykalnego: \(tokens).")
        
        let parser = Parser(tokens: tokens)
        let result = try parser.parse()
        print("Dane wyjściowe analizatora składni: \(result).")
    } catch Lexer.theError.NieprawidłowyZnak(let character) {
        print("Dane wejściowe zawierają nieprawidłowy znak: \(character).")
    } catch Parser.theError.NieoczekiwanyKoniecDanychWejściowych {
        print("Nieoczekiwany koniec danych wejściowych podczas przetwarzania.")
    } catch Parser.theError.NieprawidłowyToken(let token) {
        print("Nieprawidłowy token podczas przetwarzania: \(token).")
    } catch {
        print("Wystąpił następujący błąd: \(error).")
    }
}

evaluate(input: "10 + 3 + 5")
evaluate(input: "10 + 3 5")
