//
//  Document.swift
//  VocalTextEdit
//
//  Created by Robert Górczyński on 28/09/16.
//  Copyright © 2016 Robert Górczyński. All rights reserved.
//

import Cocoa

class Document: NSDocument {
    
    var contents: String = ""

    override class func autosavesInPlace() -> Bool {
        return true
    }

    override func makeWindowControllers() {
        // Zwraca plik Storyboard zawierający okno dokumentu.
        let storyboard = NSStoryboard(name: "Main", bundle: nil)
        let windowController = storyboard.instantiateController(withIdentifier: "Document Window Controller") as! NSWindowController
        
        let viewController = windowController.contentViewController as! ViewController
        viewController.contents = contents
        
        self.addWindowController(windowController)
    }

    override func data(ofType typeName: String) throws -> Data {
        // Miejsce na kod zapisujący dokument w postaci danych wskazanego typu. W przypadku outError != nil, należy się upewnić o utworzeniu i przypisaniu odpowiedniego błędu podczas zwrotu wartości nil.
        // Można się również zdecydować na nadpisanie fileWrapperOfType:error:, writeToURL:ofType:error: lub writeToURL:ofType:forSaveOperation:originalContentsURL:error:.
        let windowController = windowControllers[0]
        let viewController = windowController.contentViewController as! ViewController
        
        let contents = viewController.contents ?? ""
        
        if let data = contents.data(using: String.Encoding.utf8) {
            return data
        } else {
            let userInfo = [
                NSLocalizedRecoverySuggestionErrorKey:
                "Pliku nie można zapisać jako UTF-8."
            ]
            throw NSError(
                domain: "com.gorczynski.VocalTextEdit", code: 0, userInfo: userInfo)
        }
    }

    override func read(from data: Data, ofType typeName: String) throws {
        // Miejsce na kod odczytujący dokument z danych wskazanego typu. W przypadku outError != nil, należy się upewnić o utworzeniu i przypisaniu odpowiedniego błędu podczas zwrotu wartości false.
        // Można się również zdecydować na nadpisanie readFromFileWrapper:ofType:error: lub readFromURL:ofType:error:.
        // Jeżeli nadpiszesz jedną z powyższych funkcji, powinieneś nadpisać również -isEntireFileLoaded w celu zwrotu wartości false, jeśli zawartość jest wczytywana z opóźnieniem.
        if let contents = NSString(data: data, encoding: String.Encoding.utf8.rawValue) as? String {
            self.contents = contents
        } else {
            let userInfo = [
                NSLocalizedRecoverySuggestionErrorKey: "To nie jest poprawny plik UTF-8."
            ]
            throw NSError(
                domain: "com.gorczynski.VocalTextEdit", code: 0, userInfo: userInfo)
        }
    }


}

