"""
Programowanie obiektowe w Pythonie 3

Rozdział 9., Łańcuchy, serializacja i ścieżki do plików
"""
import re
from typing import Pattern, Match


def matchy(pattern: Pattern[str], text: str) -> None:
    if match := re.match(pattern, text):
        print(f"Wzorzec {pattern=!r} dopasowano na pozycji {match=!r}")
    else:
        print(f"Nie dopasowano wzorca {pattern=!r} do łańcucha {text=!r}")


# Stosunkowo proste testy możan wykonywać jako wielowierszowe łańcuchy dokumentujące.
# Aby prawidłowo obsługiwać znaki specjalne, konieczne są nieprzetworzone wielowierszone łańcuchy dokumentujące

test_matchy_incomplete = """
>>> matchy(pattern=r"sokole o", text="sokole oko")
Wzorzec pattern='sokole o' dopasowano na pozycji match=<re.Match object; span=(0, 8), match='sokole o'>
>>> matchy(pattern=r"okole oko", text="sokole oko")
Nie dopasowano wzorca pattern='okole oko' do łańcucha text='sokole oko'
"""

test_matchy_start_end = r"""
>>> matchy(pattern=r"^sokole oko$", text="sokole oko")
Wzorzec pattern='^sokole oko$' dopasowano na pozycji match=<re.Match object; span=(0, 10), match='sokole oko'>
>>> matchy(pattern=r"^sokole oko$", text="sokole ok")
Nie dopasowano wzorca pattern='^sokole oko$' do łańcucha text='sokole ok'

>>> matchy(pattern=r"\^sokole oko\$", text="sokole ok")
Nie dopasowano wzorca pattern='\\^sokole oko\\$' do łańcucha text='sokole ok'
>>> matchy(pattern=r"\^sokole oko\$", text="^sokole oko$")
Wzorzec pattern='\\^sokole oko\\$' dopasowano na pozycji match=<re.Match object; span=(0, 12), match='^sokole oko$'>

"""

test_matchy_dot = """
>>> matchy(r'sok.le oko', "sokole oko")
Wzorzec pattern='sok.le oko' dopasowano na pozycji match=<re.Match object; span=(0, 10), match='sokole oko'>
>>> matchy(r'sok.le oko', "sokxle oko")
Wzorzec pattern='sok.le oko' dopasowano na pozycji match=<re.Match object; span=(0, 10), match='sokxle oko'>
>>> matchy(r'sok.le oko', "sok le oko")
Wzorzec pattern='sok.le oko' dopasowano na pozycji match=<re.Match object; span=(0, 10), match='sok le oko'>
>>> matchy(r'sok.le oko', "sokle oko")
Nie dopasowano wzorca pattern='sok.le oko' do łańcucha text='sokle oko'
"""

test_matchy_set = """
>>> matchy(r'sok[op]le oko', "sokole oko")
Wzorzec pattern='sok[op]le oko' dopasowano na pozycji match=<re.Match object; span=(0, 10), match='sokole oko'>
>>> matchy(r'sok[op]le oko', "sokple oko")
Wzorzec pattern='sok[op]le oko' dopasowano na pozycji match=<re.Match object; span=(0, 10), match='sokple oko'>
>>> matchy(r'sok[op]le oko', "sokPle oko")
Nie dopasowano wzorca pattern='sok[op]le oko' do łańcucha text='sokPle oko'

>>> matchy(r'sokole [a-z] oko', "sokole   oko")
Nie dopasowano wzorca pattern='sokole [a-z] oko' do łańcucha text='sokole   oko'
>>> matchy(r'sokole [a-z] oko', "sokole b oko")
Wzorzec pattern='sokole [a-z] oko' dopasowano na pozycji match=<re.Match object; span=(0, 12), match='sokole b oko'>
>>> matchy(r'sokole [a-zA-Z] oko', "sokole B oko")
Wzorzec pattern='sokole [a-zA-Z] oko' dopasowano na pozycji match=<re.Match object; span=(0, 12), match='sokole B oko'>
>>> matchy(r'sokole [a-zA-Z0-9] oko', "sokole 2 oko")
Wzorzec pattern='sokole [a-zA-Z0-9] oko' dopasowano na pozycji match=<re.Match object; span=(0, 12), match='sokole 2 oko'>
"""

test_matchy_simpler_set = r"""
>>> matchy(r'\d\d\s\w\w\w\s\d\d\d\d', '26 Maj 2019')
Wzorzec pattern='\\d\\d\\s\\w\\w\\w\\s\\d\\d\\d\\d' dopasowano na pozycji match=<re.Match object; span=(0, 11), match='26 Maj 2019'>
>>> matchy(r'[0-9][0-9][ \t\n\r\f\v][A-Za-z0-9_][A-Za-z0-9_][A-Za-z0-9_][ \t\n\r\f\v][0-9][0-9][0-9][0-9]', '26 Maj 2019')
Wzorzec pattern='[0-9][0-9][ \\t\\n\\r\\f\\v][A-Za-z0-9_][A-Za-z0-9_][A-Za-z0-9_][ \\t\\n\\r\\f\\v][0-9][0-9][0-9][0-9]' dopasowano na pozycji match=<re.Match object; span=(0, 11), match='26 Maj 2019'>

"""

test_matchy_sets = r"""
>>> matchy(r'0\.[0-9][0-9]', "0.05")
Wzorzec pattern='0\\.[0-9][0-9]' dopasowano na pozycji match=<re.Match object; span=(0, 4), match='0.05'>

>>> matchy(r'0\.[0-9][0-9]', "005")
Nie dopasowano wzorca pattern='0\\.[0-9][0-9]' do łańcucha text='005'

>>> matchy(r'0\.[0-9][0-9]', "0,05")
Nie dopasowano wzorca pattern='0\\.[0-9][0-9]' do łańcucha text='0,05'


>>> matchy(r'\(abc\]', "(abc]")
Wzorzec pattern='\\(abc\\]' dopasowano na pozycji match=<re.Match object; span=(0, 5), match='(abc]'>

>>> matchy(r'\s\d\w', " 1a")
Wzorzec pattern='\\s\\d\\w' dopasowano na pozycji match=<re.Match object; span=(0, 3), match=' 1a'>

>>> matchy(r'\s\d\w', "\t5n")
Wzorzec pattern='\\s\\d\\w' dopasowano na pozycji match=<re.Match object; span=(0, 3), match='\t5n'>

>>> matchy(r'\s\d\w', " 5n")
Wzorzec pattern='\\s\\d\\w' dopasowano na pozycji match=<re.Match object; span=(0, 3), match=' 5n'>

"""

test_matchy_star = r"""
>>> matchy(r'las*o', 'lasso')
Wzorzec pattern='las*o' dopasowano na pozycji match=<re.Match object; span=(0, 5), match='lasso'>

>>> matchy(r'las*o', 'laso')
Wzorzec pattern='las*o' dopasowano na pozycji match=<re.Match object; span=(0, 4), match='laso'>

>>> matchy(r'las*o', 'lasssssso')
Wzorzec pattern='las*o' dopasowano na pozycji match=<re.Match object; span=(0, 9), match='lasssssso'>

"""

test_matchy_identifiers = r"""
>>> matchy(r'[A-Z][a-z]* [a-z]*\.', "To sekwencja.")
Wzorzec pattern='[A-Z][a-z]* [a-z]*\\.' dopasowano na pozycji match=<re.Match object; span=(0, 13), match='To sekwencja.'>
>>> matchy(r'[A-Z][a-z]* [a-z]*\.', "Nie .")
Wzorzec pattern='[A-Z][a-z]* [a-z]*\\.' dopasowano na pozycji match=<re.Match object; span=(0, 5), match='Nie .'>
>>> matchy(r'[a-z]*.*', "")
Wzorzec pattern='[a-z]*.*' dopasowano na pozycji match=<re.Match object; span=(0, 0), match=''>

"""

test_matchy_digits = r"""
>>> matchy(r'\d+\.\d+', "0.4")
Wzorzec pattern='\\d+\\.\\d+' dopasowano na pozycji match=<re.Match object; span=(0, 3), match='0.4'>
>>> matchy(r'\d+\.\d+', "1.002")
Wzorzec pattern='\\d+\\.\\d+' dopasowano na pozycji match=<re.Match object; span=(0, 5), match='1.002'>
>>> matchy(r'\d+\.\d+', "1.")
Nie dopasowano wzorca pattern='\\d+\\.\\d+' do łańcucha text='1.'

>>> matchy(r'\d?\d%', "1%")
Wzorzec pattern='\\d?\\d%' dopasowano na pozycji match=<re.Match object; span=(0, 2), match='1%'>
>>> matchy(r'\d?\d%', "99%")
Wzorzec pattern='\\d?\\d%' dopasowano na pozycji match=<re.Match object; span=(0, 3), match='99%'>
>>> matchy(r'\d?\d%', "100%")
Nie dopasowano wzorca pattern='\\d?\\d%' do łańcucha text='100%'

"""

test_matchy_explicit_count = r"""
>>> matchy(r'abc{3}', "abccc")
Wzorzec pattern='abc{3}' dopasowano na pozycji match=<re.Match object; span=(0, 5), match='abccc'>
>>> matchy(r'(abc){3}', "abccc")
Nie dopasowano wzorca pattern='(abc){3}' do łańcucha text='abccc'
>>> matchy(r'(abc){3}', "abcabcabc")
Wzorzec pattern='(abc){3}' dopasowano na pozycji match=<re.Match object; span=(0, 9), match='abcabcabc'>

"""

test_matchy_escape_dot = r"""
>>> matchy(r'[A-Z][a-z]*( [a-z]+)*\.$', "Jedz.")
Wzorzec pattern='[A-Z][a-z]*( [a-z]+)*\\.$' dopasowano na pozycji match=<re.Match object; span=(0, 5), match='Jedz.'>
>>> matchy(r'[A-Z][a-z]*( [a-z]+)*\.$', "Jem sporo warzyw.")
Wzorzec pattern='[A-Z][a-z]*( [a-z]+)*\\.$' dopasowano na pozycji match=<re.Match object; span=(0, 17), match='Jem sporo warzyw.'>
>>> matchy(r'[A-Z][a-z]*( [a-z]+)*\.$', "Bardzo dobry obiad.")
Wzorzec pattern='[A-Z][a-z]*( [a-z]+)*\\.$' dopasowano na pozycji match=<re.Match object; span=(0, 19), match='Bardzo dobry obiad.'>

"""

from typing import Optional


def email_domain(text: str) -> Optional[str]:
    email_pattern = r"[a-z0-9._%+-]+@([a-z0-9.-]+\.[a-z]{2,})"
    if match := re.match(email_pattern, text, re.IGNORECASE):
        return match.group(1)
    else:
        return None


test_email_domain = """
>>> email_domain("wodzinio@supersajt.pl")
'supersajt.pl'

"""


def email_domain_2(text: str) -> Optional[str]:
    email_pattern = r"(?P<name>[a-z0-9._%+-]+)@(?P<domain>[a-z0-9.-]+\.[a-z]{2,})"
    if match := re.match(email_pattern, text, re.IGNORECASE):
        return match.groupdict()["domain"]
    else:
        return None


test_email_domain_2 = """
>>> email_domain_2("wodzinio@supersajt.pl")
'supersajt.pl'

"""

__test__ = {name: case for name, case in globals().items() if name.startswith("test_")}
