"""
Programowanie obiektowe w Pythonie 3

Rozdział 2., Obiekty w Pythonie
"""

import math


class Point:
    """
    Reprezentuje punkt w geometrycznej przestrzeni dwuwymiarowej.

    >>> p_0 = Point()
    >>> p_1 = Point(3, 4)
    >>> p_0.calculate_distance(p_1)
    5.0
    """

    def __init__(self, x: float = 0, y: float = 0) -> None:
        """
        Metoda inicjalizuje nowy punkt i umieszcza go w określonym miejscu.
        Można podać współrzędne x i y tego punktu. Jeśli zostaną pominięte,
        punkt zostanie umieszczony w początku układu współrzędnych.

        :param x: float współrzędna x
        :param y: float współrzędna y
        """
        self.move(x, y)

    def move(self, x: float, y: float) -> None:
        """
        Metoda przenosi punkt w nowe miejsce w przestrzeni dwuwymiarowej.

        :param x: float współrzędna x
        :param y: float współrzędna y
        """
        self.x = x
        self.y = y

    def reset(self) -> None:
        """
        Resetuje położenie punktu przenosząc go do początku układu 
        współrzędnych (punktu 0, 0).
        """
        self.move(0, 0)

    def calculate_distance(self, other: "Point") -> float:
        """
        Metoda oblicza odległość euklidesową danego punktu oraz punktu
        przekazanego jako parametr.

        :param other: Point drugi punkt
        :return: float odległość między punktami
        """
        return math.hypot(self.x - other.x, self.y - other.y)


test_point_help = """
>>> help(Point)
Help on class Point in module point_4:
<BLANKLINE>
class Point(builtins.object)
 |  Point(x: float = 0, y: float = 0) -> None
 |  
 |  Reprezentuje punkt w geometrycznej przestrzeni dwuwymiarowej.
 |  
 |  >>> p_0 = Point()
 |  >>> p_1 = Point(3, 4)
 |  >>> p_0.calculate_distance(p_1)
 |  5.0
 |  
 |  Methods defined here:
 |  
 |  __init__(self, x: float = 0, y: float = 0) -> None
 |      Metoda inicjalizuje nowy punkt i umieszcza go w określonym miejscu.
 |      Można podać współrzędne x i y tego punktu. Jeśli zostaną pominięte,
 |      punkt zostanie umieszczony w początku układu współrzędnych.
 |      
 |      :param x: float współrzędna x
 |      :param y: float współrzędna y
 |  
 |  calculate_distance(self, other: 'Point') -> float
 |      Metoda oblicza odległość euklidesową danego punktu oraz punktu
 |      przekazanego jako parametr.
 |      
 |      :param other: Point drugi punkt
 |      :return: float odległość między punktami
 |  
 |  move(self, x: float, y: float) -> None
 |      Metoda przenosi punkt w nowe miejsce w przestrzeni dwuwymiarowej.
 |      
 |      :param x: float współrzędna x
 |      :param y: float współrzędna y
 |  
 |  reset(self) -> None
 |      Resetuje położenie punktu przenosząc go do początku układu 
 |      współrzędnych (punktu 0, 0).
 |  
 |  ----------------------------------------------------------------------
 |  Data descriptors defined here:
 |  
 |  __dict__
 |      dictionary for instance variables (if defined)
 |  
 |  __weakref__
 |      list of weak references to the object (if defined)
<BLANKLINE>

"""

test_main = """
>>> main()
p1.calculate_distance(p2)=5.0
"""

__test__ = {name: case for name, case in globals().items() if name.startswith("test_")}


def main() -> None:
    """
    Does the useful work.

    >>> main()
    p1.calculate_distance(p2)=5.0
    """
    p1 = Point()
    p2 = Point(3, 4)
    print(f"{p1.calculate_distance(p2)=}")


if __name__ == "__main__":
    main()
