"""
Programowanie obiektowe w Pythonie 3

Rozdział 3, Kiedy obiekty są do siebie podobne
"""
from __future__ import annotations
from typing import Optional, Protocol

## Rozszerzanie typów wbudowanych

class ContactList(list["Contact"]):
    def search(self, name: str) -> list["Contact"]:
        """Wszystkie kontakty o imieniu zawierającym podany łańcuch"""
        matching_contacts: list["Contact"] = []
        for contact in self:
            if name in contact.name:
                matching_contacts.append(contact)
        return matching_contacts


class Contact:
    all_contacts = ContactList()

    def __init__(self, name: str, email: str) -> None:
        self.name = name
        self.email = email
        Contact.all_contacts.append(self)

    def __repr__(self) -> str:
        return (
            f"{self.__class__.__name__}(" 
            f"{self.name!r}, {self.email!r}" 
            f")"
        )


class Supplier(Contact):
    def order(self, order: "Order") -> None:
        print(
            "W tym miejscu prawdziwy system wysłałby "
            f"zamówienie '{order}' do '{self.name}'"
        )


class Order:
    pass


test_search = """
>>> Contact.all_contacts = ContactList()

>>> c1 = Contact("Janek A", "janeka@mojstajt.com.pl")
>>> c2 = Contact("Janek B", "janekb@mojstajt.com.pl")  
>>> c3 = Contact("Janka C", "jankac@mojstajt.com.pl")  
>>> [c.name for c in Contact.all_contacts.search("Janek")]
['Janek A', 'Janek B']
"""


class LongNameDict(dict[str, int]):
    def longest_key(self) -> Optional[str]:
        """To samo co max(self, key=len), ale bardziej zrozumiałe"""
        longest = None
        for key in self:
            if longest is None or len(key) > len(longest):
                longest = key
        return longest


test_longnamedict = """
>>> articles_read = LongNameDict()
>>> articles_read['Lucyna'] = 42
>>> articles_read['zimny_leszek'] = 6 
>>> articles_read['Staszek'] = 7
>>> articles_read.longest_key() 
'zimny_leszek'
>>> max(articles_read, key=len)
'zimny_leszek'
"""

## Przesłanienie i super


class Friend_1(Contact):
    def __init__(self, name: str, email: str, phone: str) -> None:
        super().__init__(name, email)
        self.phone = phone


test_friend_1 = """
>>> f = Friend_1("Darek", "darek@kumplepogrob.org.pl", "678 411 214")
>>> f
Friend_1('Darek', 'darek@kumplepogrob.org.pl')
"""


class Emailable(Protocol):
    email: str


class MailSender(Emailable):
    def send_mail(self, message: str) -> None:
        print(f"Wysyłam wiadomość e-mail do {self.email=}")
        # Implementacja logiki wysyłania wiadomości


class EmailableContact(Contact, MailSender):
    pass


test_emailable_contact = """
>>> Contact.all_contacts = ContactList()

>>> e = EmailableContact("Janek B", "janekb@superowo.pl")
>>> Contact.all_contacts
[EmailableContact('Janek B', 'janekb@superowo.pl')] 
>>> e.send_mail("Testujemy wysyłanie e-maili")
Wysyłam wiadomość e-mail do self.email='janekb@superowo.pl'
"""

## Wielokrotne dziedziczenie


class AddressHolder:
    def __init__(self, street: str, city: str, state: str, code: str) -> None:
        self.street = street
        self.city = city
        self.state = state
        self.code = code


class Friend(Contact, AddressHolder):
    def __init__(
        self,
        name: str,
        email: str,
        phone: str,
        street: str,
        city: str,
        state: str,
        code: str,
    ) -> None:
        Contact.__init__(self, name, email)
        AddressHolder.__init__(self, street, city, state, code)
        self.phone = phone


test_naive_friend = """
>>> norek = Friend("Norek", "norek@mojsajt.com.pl", "662 343 342", "Seriali Komediowych 15", "Warszawa", "Mazowieckie", "00-012")
>>> norek.phone
'662 343 342'

"""


## Problematyczny diament

class BaseClass:
    num_base_calls = 0

    def call_me(self) -> None:
        print("Wywołanie metody klasy BaseClass")
        self.num_base_calls += 1


class LeftSubclass(BaseClass):
    num_left_calls = 0

    def call_me(self) -> None:
        BaseClass.call_me(self)
        print("Wywołanie metody klasy LeftSubclass")
        self.num_left_calls += 1


class RightSubclass(BaseClass):
    num_right_calls = 0

    def call_me(self) -> None:
        BaseClass.call_me(self)
        print("Wywołanie metody klasy RightSubclass")
        self.num_right_calls += 1


class Subclass(LeftSubclass, RightSubclass):
    num_sub_calls = 0

    def call_me(self) -> None:
        LeftSubclass.call_me(self)
        RightSubclass.call_me(self)
        print("Wywołanie metody klasy Subclass")
        self.num_sub_calls += 1

test_diamond = """
>>> s = Subclass()
>>> s.call_me()
Wywołanie metody klasy BaseClass
Wywołanie metody klasy LeftSubclass
Wywołanie metody klasy BaseClass
Wywołanie metody klasy RightSubclass
Wywołanie metody klasy Subclass
>>> print(
... s.num_sub_calls,
... s.num_left_calls,
... s.num_right_calls,
... s.num_base_calls)
1 1 1 2
"""


class LeftSubclass_S(BaseClass):
    num_left_calls = 0

    def call_me(self) -> None:
        super().call_me()
        print("Wywołanie metody klasy LeftSubclass_S")
        self.num_left_calls += 1


class RightSubclass_S(BaseClass):
    num_right_calls = 0

    def call_me(self) -> None:
        super().call_me()
        print("Wywołanie metody klasy RightSubclass_S")
        self.num_right_calls += 1


class Subclass_S(LeftSubclass_S, RightSubclass_S):
    num_sub_calls = 0

    def call_me(self) -> None:
        super().call_me()
        print("Wywołanie metody klasy Subclass_S")
        self.num_sub_calls += 1


test_super_diamond = """
>>> ss = Subclass_S()
>>> ss.call_me()
Wywołanie metody klasy BaseClass
Wywołanie metody klasy RightSubclass_S
Wywołanie metody klasy LeftSubclass_S
Wywołanie metody klasy Subclass_S
>>> print(
... ss.num_sub_calls,
... ss.num_left_calls,
... ss.num_right_calls,
... ss.num_base_calls)
1 1 1 1

>>> from pprint import pprint
>>> pprint(Subclass_S.__mro__)
(<class 'commerce_naive.Subclass_S'>,
 <class 'commerce_naive.LeftSubclass_S'>,
 <class 'commerce_naive.RightSubclass_S'>,
 <class 'commerce_naive.BaseClass'>,
 <class 'object'>)
"""


__test__ = {name: case for name, case in globals().items() if name.startswith("test_")}
