"""
Programowanie obiektowe w Pythonie 3

Rozdział 8., Łączenie programowania obiektowego i funkcyjnego
"""
from __future__ import annotations
import heapq
import time
from typing import Any, Optional, cast, Callable
from dataclasses import dataclass, field

Callback = Callable[[int], None]


@dataclass(frozen=True, order=True)
class Task:
    """
    Definicja zadania, które ma być wykonane po upłynięciu pewego czasu od uruchomienia
    obiektu Scheduler. Podany czas jest interpretowany przez klasę Scheduler, 
    zazwyczaj jest podawany w sekundach.
    Zadaniejest dowolną funkcję, z którą zostanie skojarzony czas, który ma upłynąć 
    nim zostanie wywołana.

    Jeśli opóźnienie zostało podane, to się powtarza. Limit określa ile razy 
    wykonanie zadania może zostać powtórzone, domyślnie zostanie ono wykonane tylko raz.
    """

    scheduled: int
    callback: Callback = field(compare=False)
    delay: int = field(default=0, compare=False)
    limit: int = field(default=1, compare=False)

    def repeat(self, current_time: int) -> Optional["Task"]:
        """
        Jeśli limit > 1, to zadanie może zostać wykonane ponownie.
        Tworzymy nowy obiekt Task z godziną o której powinno ono zostać ponownie wykonane
        """
        if self.delay > 0 and self.limit > 2:
            return Task(
                current_time + self.delay,
                cast(Callback, self.callback),  # type: ignore [misc]
                self.delay,
                self.limit - 1,
            )
        elif self.delay > 0 and self.limit == 2:
            return Task(
                current_time + self.delay,
                cast(Callback, self.callback),  # type: ignore [misc]
            )
        else:
            return None


class Scheduler:
    """
    Planowanie zadań do wykonania.
    Sposób użycia :meth:`enter` by dodać zadanie do kolejki.
    Sposób użycia :meth:`run` by rozpocząć przetwarzanie.
    Obecnie używane jest wywołanie time.sleep() co oznacza, że ``after`` i ``delay`` są wyrażone w sekundach.
    """

    def __init__(self) -> None:
        self.tasks: list[Task] = []

    def enter(
        self,
        after: int,
        task: Callback,
        delay: int = 0,
        limit: int = 1,
    ) -> None:
        new_task = Task(after, task, delay, limit)
        heapq.heappush(self.tasks, new_task)

    def run(self) -> None:
        current_time = 0
        while self.tasks:
            next_task = heapq.heappop(self.tasks)
            if (delay := next_task.scheduled - current_time) > 0:
                time.sleep(next_task.scheduled - current_time)
            current_time = next_task.scheduled
            next_task.callback(current_time)  # type: ignore [misc]
            if again := next_task.repeat(current_time):
                heapq.heappush(self.tasks, again)


import datetime


def format_time(message: str) -> None:
    now = datetime.datetime.now()
    print(f"{now:%I:%M:%S}: {message}")


def one(timer: float) -> None:
    format_time("Wywołano funkcję 'one'")


def two(timer: float) -> None:
    format_time("Wywołano funkcję 'two'")


def three(timer: float) -> None:
    format_time("Wywołano funkcję 'three'")


class Repeater:
    def __init__(self) -> None:
        self.count = 0

    def four(self, timer: float) -> None:
        self.count += 1
        format_time(f"Wywołano funkcję 'four': {self.count}")


class Repeater_2:
    def __init__(self) -> None:
        self.count = 0
    
    def __call__(self, timer: float) -> None:
        self.count += 1
        format_time(f"Wywołano funkcję 'four': {self.count}")


test_workers = """
>>> from unittest.mock import Mock, patch
>>> expected_date = datetime.datetime(2019, 10, 26, 11, 12, 13)
>>> with patch('datetime.datetime', now=Mock(return_value=expected_date)):
...     one(42)
...     two(42)
...     three(55)
11:12:13: Wywołano funkcję 'one'
11:12:13: Wywołano funkcję 'two'
11:12:13: Wywołano funkcję 'three'

>>> rpt = Repeater()
>>> with patch('datetime.datetime', now=Mock(return_value=expected_date)):
...     rpt.four(42)
...     rpt.four(43)
...     rpt.four(44)
11:12:13: Wywołano funkcję 'four': 1
11:12:13: Wywołano funkcję 'four': 2
11:12:13: Wywołano funkcję 'four': 3

>>> rpt2 = Repeater_2()
>>> with patch('datetime.datetime', now=Mock(return_value=expected_date)):
...     rpt2(42)
...     rpt2(43)
...     rpt2(44)
11:12:13: Wywołano funkcję 'four': 1
11:12:13: Wywołano funkcję 'four': 2
11:12:13: Wywołano funkcję 'four': 3

"""


__test__ = {name: case for name, case in globals().items() if name.startswith("test_")}

if __name__ == "__main__":
    s = Scheduler()
    s.enter(1, one)
    s.enter(2, one)
    s.enter(2, two)
    s.enter(4, two)
    s.enter(3, three)
    s.enter(6, three)
    repeater = Repeater()
    s.enter(5, repeater.four, delay=1, limit=5)
    s.run()

    s2 = Scheduler()
    s2.enter(5, Repeater_2(), delay=1, limit=5)
    s2.run()
