"""
Programowanie obiektowe w Pythonie 3

Rozdział 6., Abstrakcyjne klasy bazowe i przeciążanie operatorów
"""
from __future__ import annotations
from contextlib import AbstractContextManager
from io import StringIO
import sys
from typing import Type, Literal, Optional
from types import TracebackType


if sys.version_info >= (3, 9):
    # Python 3.9 with revised generics
    class DebuggingOnly(AbstractContextManager["DebuggingOnly"]):
        """Podobna do contextlib.redirect_stdout"""

        def __enter__(self) -> "DebuggingOnly":
            self.previous = sys.stdout
            self.buffer = StringIO()
            sys.stdout = self.buffer
            return self

        def __exit__(
            self,
            exc_class: Optional[Type[BaseException]],
            exc: Optional[BaseException],
            exc_tb: Optional[TracebackType],
        ) -> Literal[False]:
            sys.stdout = self.previous
            if exc:
                print(f"--EX-->{exc!r}")
                for line in self.buffer.getvalue().splitlines():
                    print(f"       {line}")
            return False


else:
    # Tego oczekiwał Python 3.8.
    from typing import ContextManager

    class DebuggingOnly(ContextManager["DebuggingOnly"]):
        """Podobna do contextlib.redirect_stdout"""

        def __enter__(self) -> "DebuggingOnly":
            self.previous = sys.stdout
            self.buffer = StringIO()
            sys.stdout = self.buffer
            return self

        def __exit__(
            self,
            exc_class: Optional[Type[BaseException]],
            exc: Optional[BaseException],
            exc_tb: Optional[TracebackType],
        ) -> Literal[False]:
            sys.stdout = self.previous
            if exc:
                print(f"--EX-->{exc!r}")
                for line in self.buffer.getvalue().splitlines():
                    print(f"       {line}")
            return False
