"""
Programowanie obiektowe w Pythonie 3

Rozdział 11., Często stosowane wzorce projektowe
"""
from __future__ import annotations
import random
import json
import time
from dice import Dice
from typing import List, Protocol


class Observer(Protocol):
    def __call__(self) -> None:
        ...


class Observable:
    def __init__(self) -> None:
        self._observers: list[Observer] = []

    def attach(self, observer: Observer) -> None:
        self._observers.append(observer)

    def detach(self, observer: Observer) -> None:
        self._observers.remove(observer)

    def _notify_observers(self) -> None:
        for observer in self._observers:
            observer()


Hand = List[int]


class ZonkHandHistory(Observable):
    def __init__(self, player: str, dice_set: Dice) -> None:
        super().__init__()
        self.player = player
        self.dice_set = dice_set
        self.rolls: list[Hand]

    def start(self) -> Hand:
        self.dice_set.roll()
        self.rolls = [self.dice_set.dice]
        self._notify_observers()  # Zmiana stanu
        return self.dice_set.dice

    def roll(self) -> Hand:
        self.dice_set.roll()
        self.rolls.append(self.dice_set.dice)
        self._notify_observers()  # Zmiana stanu
        return self.dice_set.dice


class SaveZonkHand(Observer):
    def __init__(self, hand: ZonkHandHistory) -> None:
        self.hand = hand
        self.count = 0

    def __call__(self) -> None:
        self.count += 1
        message = {
            "gracz": self.hand.player,
            "rzut": self.count,
            "ręka": json.dumps(self.hand.rolls),
            "czas": time.time(),
        }
        print(f"SaveZonkHand {message}")


class ThreePairZonkHand:
    """Obserwator dla klasy ZonkHandHistory"""

    def __init__(self, hand: ZonkHandHistory) -> None:
        self.hand = hand
        self.zonked = False

    def __call__(self) -> None:
        last_roll = self.hand.rolls[-1]
        distinct_values = set(last_roll)
        self.zonked = len(distinct_values) == 3 and all(
            last_roll.count(v) == 2 for v in distinct_values
        )
        if self.zonked:
            print("Zonk trzech par!")


test_hand_history = """
>>> from unittest.mock import Mock, call
>>> mock_observer = Mock()
>>> import random
>>> random.seed(42)

>>> d = Dice.from_text("6d6")
>>> player = ZonkHandHistory("Bo", d)
>>> player.attach(mock_observer)
>>> player.start()
[1, 1, 2, 3, 6, 6]

>>> player.roll()
[1, 2, 2, 6, 6, 6]

>>> player.rolls
[[1, 1, 2, 3, 6, 6], [1, 2, 2, 6, 6, 6]]

>>> mock_observer.mock_calls
[call(), call()]
"""

test_zonk_observer = """
>>> import random
>>> random.seed(42)

>>> d = Dice.from_text("6d6")
>>> player = ZonkHandHistory("Bo", d)

>>> save_history = SaveZonkHand(player)
>>> player.attach(save_history)
>>> r1 = player.start()
SaveZonkHand {'gracz': 'Bo', 'rzut': 1, 'ręka': '[[1, 1, 2, 3, 6, 6]]', 'czas': ...}
>>> r1
[1, 1, 2, 3, 6, 6]

>>> r2 = player.roll()
SaveZonkHand {'gracz': 'Bo', 'rzut': 2, 'ręka': '[[1, 1, 2, 3, 6, 6], [1, 2, 2, 6, 6, 6]]', 'czas': ...}
>>> r2
[1, 2, 2, 6, 6, 6]

>>> player.rolls
[[1, 1, 2, 3, 6, 6], [1, 2, 2, 6, 6, 6]]

"""

test_zonk_observer_2 = """
>>> import random
>>> random.seed(21)

>>> d = Dice.from_text("6d6")
>>> player = ZonkHandHistory("Darek", d)

>>> save_history = SaveZonkHand(player)
>>> player.attach(save_history)
>>> find_3_pair = ThreePairZonkHand(player)
>>> player.attach(find_3_pair)

>>> r1 = player.start()
SaveZonkHand {'gracz': 'Darek', 'rzut': 1, 'ręka': '[[2, 3, 4, 4, 6, 6]]', 'czas': ...}
>>> r1
[2, 3, 4, 4, 6, 6]

>>> r2 = player.roll()
SaveZonkHand {'gracz': 'Darek', 'rzut': 2, 'ręka': '[[2, 3, 4, 4, 6, 6], [2, 2, 4, 4, 5, 5]]', 'czas': ...}
Zonk trzech par!
>>> r2
[2, 2, 4, 4, 5, 5]

>>> player.rolls
[[2, 3, 4, 4, 6, 6], [2, 2, 4, 4, 5, 5]]

"""


def find_seed() -> None:
    d = Dice.from_text("6d6")
    player = ZonkHandHistory("Darek", d)

    find_3_pair = ThreePairZonkHand(player)
    player.attach(find_3_pair)

    for s in range(10_000):
        random.seed(s)
        player.start()
        if find_3_pair.zonked:
            print(f"Wartość {s}, rzuty: {player.rolls} ")
        player.roll()
        if find_3_pair.zonked:
            print(f"Wartość {s}, rzuty: {player.rolls} ")


__test__ = {name: case for name, case in globals().items() if name.startswith("test_")}

if __name__ == "__main__":
    find_seed()
