
 * Główna klasa obsługująca cykl życia gry. Inicjalizuje pozostałe komponenty, takie jak 
 * Board czy BoardModel. Nasłuchuje także zdarzenia DOM i przekształca kliknięcia na współrzędne.
 * @param canvas obiekt płótna używany do rysowania
 */
function OnlineGame(canvas, socket, endGameFn) {
	this._boardRect = null;
	this._canvas = canvas;
	this._ctx = canvas.getContext("2d");
	this._boardModel = new BoardModel();

	this._boardRenderer = new BoardRenderer(this._ctx, this._boardModel);
	this.handleResize();
    
    this._socket = socket;
    this._endGameFn = endGameFn;

    socket.on("turn", (function(turn) {
        console.log("Twoja kolej");
        this._makeTurn(turn.x);
    }).bind(this));

    socket.on("error", function(error) {
        alert(error.cause);
    });
}

_p = OnlineGame.prototype;

/** 
* Obsługuje kliknięcie (lub stuknięcie) na płótnie. Przekształca współrzędne płótna na kolumnę planszy i wykonuje ruch w ramach tej kolumny 
* @param x — współrzędna x stuknięcia lub kliknięcia
* @param y — współrzędna y stuknięcia lub kliknięcia
*/ 

_p.handleClick = function(x, y) {
	// Pobierz indeks kolumny
	var column = Math.floor((x - this._boardRect.x)/this._boardRect.cellSize);

	
    if (this._boardModel.isTurnValid(column)) {
        // Nie umieszczaj jeszcze pionka — wyślij ruch do serwera 
        this._requestTurn(column);
    } else {
        alert("Nieprawidłowy ruch");
        // Zignoruj ruch
    }
};

/** 
* Poinformuj serwer o wykonaniu ruchu. Nie umieszczaj pionków na planszy w tym momencie 
* @param column — kolumna, do której wrzucamy pionek
*/ 

_p._requestTurn = function(column) {
    this._socket.emit("turn", column);
};

/** 
* Wywoływane w momencie przesłania odpowiedzi przez serwer. Umieszczamy pionek i sprawdzamy wynik. Jeśli gra się zakończyła — wracamy do lobby
* @param column — kolumna, do której został wrzucony pionek
*/ 

_p._makeTurn = function(column) {
    // Wykonaj ruch i sprawdź wynik
    var turn = this._boardModel.makeTurn(column);

    // Jeśli ruch jest poprawny, odśwież planszę i narysuj nowy pionek
    if (turn.status == BoardModel.ILLEGAL_TURN) {
        alert("Ups, nie jesteśmy zsynchronizowani z serwerem!"); 
        return;
    }

    this._boardRenderer.drawToken(turn.x, turn.y);

    // Czy po ostatnim ruchu mamy zwycięzcę? 
    if (turn.status != BoardModel.NONE) {
        this._notifyAboutGameEnd(turn);
        this._reset();
        this._endGameFn();
    }
};

/**
 * Przywróc _boardModel i odrysuj planszę
 */
_p._reset = function() {
	this._clearCanvas();
	this._boardModel.reset();
	this._boardRenderer.repaint();
};

/**
 * Wywoływane przy zmianie rozmiaru ekranu. Trzeba obliczyć nowy rozmiar i położenie planszy.
 */
_p.handleResize = function() {
    this._clearCanvas();
    this._boardRect = this._getBoardRect();
	this._boardRenderer.setSize(this._boardRect.x, this._boardRect.y, this._boardRect.cellSize);
	this._boardRenderer.repaint();
};

/**
 * Oblicz optymalne położenie i rozmiar planszy
 * @return obiekt zawierający rozmiar i położenie planszy:
 * {
 * 		x: współrzędna x lewego górnego wierzchołka planszy
 * 		y: współrzędna y lewego górnego wierzchołka planszy
 * 		cellSize: optymalny rozmiar planszy (w pikselach)
 * }
 */
_p._getBoardRect = function() {
    var cols = this._boardModel.getCols();
    var rows = this._boardModel.getRows();
	var cellSize = Math.floor(
			Math.min(this._canvas.width/cols, this._canvas.height/rows));
	
	var boardWidth = cellSize*cols;
	var boardHeight = cellSize*rows;

	return {
		x: Math.floor((this._canvas.width - boardWidth)/2),
		y: Math.floor((this._canvas.height - boardHeight)/2),
		cellSize: cellSize
	}
};

/**
 * Czyści tło planszy.
 */
_p._clearCanvas = function() {
    this._ctx.fillStyle = "white";
    this._ctx.fillRect(0, 0, this._canvas.width, this._canvas.height);
};

/**
 * Przedstawia komunikat w momencie zakończenia gry
 * @param turn ostatnia kolejka
 */
_p._notifyAboutGameEnd = function(turn) {
    if (turn.status == BoardModel.WIN) {
        // Poinformuj o wyniku gry i zresetuj grę
        alert((turn.piece == BoardModel.RED ? "red" : "green") + " wygrał mecz!");
    }

    // Analogicznie w przypadku remisu
    if (turn.status == BoardModel.DRAW) {
        alert("To jest remis!");
    }
};