exports = module.exports = GameSession;

var board = require("./public/js/BoardModel");
var BoardModel = board.BoardModel;

/**
 * Instancja tej klasy jest tworzona, gdy dwaj gracze zgadzają się na grę.
 * Klasa tworzy odrębny pokój dla graczy i ustawia dodatkowe funkcje nasłuchujące.
 * Po zakończeniu gry funkcje są wyrejestrowane, a pokój usuwany.
 * Po rozpoczęciu rozgrywki jest tworzona nowa instancja klasy GameSession, z nowym ID i nowym pokojem
 *
 * @param id unikalny ID sesji, powinien być generowany zewnętrznie. Jest używany jako nazwa pokoju.
 * @param player1 pierwszy gracz
 * @param player2 drugi gracz
 * @param onEndGame funkcja, która zostanie wywołana po zakończeniu gry.
 */
function GameSession(id, player1, player2, onEndGame) {
    this._roomName = "game" + id;
    this._players = [player1, player2];
    this._currentPlayer = 0;

    this._boardModel = new BoardModel();
    this._onEndGame = onEndGame;

    for (var i = 0; i < this._players.length; i++) {
        this._setupGameListeners(i);
    }
}

var _p = GameSession.prototype;

_p._setupGameListeners = function(playerIndex) {
    var socket = this._players[playerIndex].socket;
    socket.join(this._roomName);

    socket.on("turn", (function(column) {
        if (playerIndex != this._currentPlayer) {
            // Z jakiejś przyczyny niewłaściwy gracz próbuje wykonać ruch
            socket.emit("error", {
                cause: "W tej chwili nie jest Twoja kolej"
            });
            return;
        }

        // Spróbujmy wykonac ruch i sprawdźmy, co się stanie
        var turn = this._boardModel.makeTurn(column);

        // Ruch jest nieprawidłowy?
        if (turn.status == BoardModel.ILLEGAL_TURN) {
            socket.emit("error", {
                cause: "Ten ruch jest nieprawidłowy"
            });
            return;
        }

        // Ruch jest prawidłowy, dlatego możemy wysłać go do obu uczestników
        socket.manager.sockets.to(this._roomName).emit("turn", turn);

        // Następny gracz staje się aktywny 
        this._currentPlayer = (this._currentPlayer + 1)%2;

        // Jeśli został spełniony warunek zwycięstwa lub remisu, gracze zostają przeniesieni do lobby. Wyczyść słuchacze zdarzeń
        if (turn.status == BoardModel.WIN || turn.status == BoardModel.DRAW) {
            // Zakończ grę, opuść pokój i wyrejestruj słuchacze
            for (var i = 0; i < this._players.length; i++) {
                this._players[i].socket.removeAllListeners("turn");
                this._players[i].socket.leave(this._roomName);
            }
        }

        // Wykonaj wywołanie zwrotne
        this._onEndGame();
    }).bind(this));
};