/**
 * Kontrolka, która symuluje dżojstik.
 * @param canvas płótno będące źródłem zdarzeń DOM
 */
function Joystick(canvas) {
    EventEmitter.call(this);

    this._ctx = canvas.getContext("2d");

    // Pozycja dżojstika w ramach płótna
    this._x = 0;
    this._y = 0;
    this._controllerRadius = 100;

    // Aktualne kąt i promień
    this._azimuth = 0;
    this._radius = 0;

    // Aktualne położenie drążka w odniesieniu do środka kontrolki
    this._deltaX = 0;
    this._deltaY = 0;

    var input = new InputHandler(canvas);

    // Dżojstik to esencja ruchu - nie musimy wprowadzać żadnego progu ruchu.
    input.setMoveThreshold(0);

    // Powiąż zdarzenia DOM
    var bindedDownOrMove = this._onDownOrMove.bind(this);
    input.on("move", bindedDownOrMove);
    input.on("down", bindedDownOrMove);
    input.on("up", this._onUp.bind(this));
}

extend(Joystick, EventEmitter);

_p = Joystick.prototype;

_p.getPosition = function() {
    return {
        x: this._x,
        y: this._y
    }
};

_p.setPosition = function(x, y) {
    this._x = x;
    this._y = y;
};


_p.getControllerRadius = function() {
    return this._controllerRadius;
};

_p.setControllerRadius = function(radius) {
    this._controllerRadius = radius;
};

_p.getRadius = function() {
    return this._radius;
};

_p.getAzimuth = function() {
    return this._azimuth;
};

/**
 * Renderuj interfejs do obsługi dżojstika
 */
_p.draw = function() {
    var ctx = this._ctx;
    ctx.fillStyle = "lightgreen";
    ctx.strokeStyle = "darkgray";
    ctx.lineWidth = 5;

    ctx.beginPath();
    ctx.arc(this._x, this._y, this._controllerRadius, 0, Math.PI*2, false);
    ctx.fill();
    ctx.stroke();

    ctx.fillStyle = "lightblue";
    ctx.beginPath();
    ctx.arc(this._x + this._deltaX,
        this._y + this._deltaY, 30, 0, Math.PI*2, false);
    ctx.fill();
    ctx.stroke();

};

_p._onDownOrMove = function(coords) {
    // Po otrzymaniu zdarzeń move lub down zapisujemy aktualne
    // wartości zmian, a następnie zmieniamy kąt i promień.

    var deltaX = coords.x - this._x;
    var deltaY = coords.y - this._y;
    this._updateJoystickValues(deltaX, deltaY);
};

_p._onUp = function(x, y) {
    // Jeśli nie ma interakcji, przywróć stan początkowy.
    this._updateJoystickValues(0, 0);
};

_p._updateJoystickValues = function(deltaX, deltaY) {
    var newAzimuth = 0;
    var newRadius = 0;

    // Jeśli stan dżojstika się nie zmienił, nie musimy niczego robić.
    if (deltaX != 0 || deltaY != 0) {
        newRadius = Math.sqrt(deltaX*deltaX + deltaY*deltaY)/this._controllerRadius;

        // Użytkownik przesunął drążek za daleko — przywracamy stan początkowy.
        if (newRadius > 1) {
            deltaX = 0;
            deltaY = 0;
            newRadius = 0;
        } else {
            newAzimuth = Math.atan2(deltaY, deltaX);
        }
    }

    // Jeśli wartości uległy zmianie, poinformuj słuchaczy.
    if (this._azimuth != newAzimuth || this._radius != newRadius) {
        this._azimuth = newAzimuth;
        this._radius = newRadius;
        this._deltaX = deltaX;
        this._deltaY = deltaY;
        this.emit("joystickchange", {
            azimuth: newAzimuth,
            radius: newRadius });
    }
};