var express = require("express");
var http = require("http");
var GameSession = require("./GameSession");

var app = express();
var server = http.createServer(app);
var io = require("socket.io").listen(server, {
    "polling duration": 10
});

app.set("views", __dirname + "/views");
app.set("view engine", "ejs");

app.use(express.bodyParser());
app.use(express.cookieParser());
app.use(express.session({ secret: "gameserversession" }));
app.use(app.router);
app.use(express.static(__dirname + "/public"));
app.use(express.errorHandler());

app.get("/", function(req, res) {
    res.sendfile(__dirname + "/public/index.html"); 
});

io.sockets.on("connection", onConnection);
server.listen(80);

/** 
* Globalne liczniki i rejestr użytkowników 
*/ 

var maxUserId = 0;
var maxGameId = 0;
var users = {};

/**
 * Funkcja obsługi połączenia - główna funkcja serwera dla wielu graczy.
 * @param socket obiekt gniazda
 */
function onConnection(socket) {
    var userId = addUser(socket);

    socket.on('challenge', function (challengedUserId, respond) {
        if (userId == challengedUserId) {
            respond({ error: "Nie można wyzwać samego siebie" });
        } else if (!users[challengedUserId]) {
            respond({ error: "Nie udało się znaleźć użytkownika " + challengedUserId });
        } else if (users[challengedUserId].status != "ready") {
            respond({ error: "Użytkownik nie jest gotowy do gry" });
        } else {
            startGame(users[userId], users[challengedUserId], respond)
        }
    });

    socket.on('disconnect', function () {
        console.log(users[userId].name + " rozłączył się");
        socket.broadcast.emit('user-left', users[userId]);
        delete users[userId];
    });
}


/** 
* Rejestruje użytkownika na serwerze, informując pozostałych klientów o zaistniałej sytuacji. Ponadto
* nowo zarejestrowany użytkownik otrzymuje listę wszystkich aktywnych klientów. 
* @param socket obiekt gniazda podłączonego klienta 
* @returns ID nowo utworzonego użytkownika  
*/ 

function addUser(socket) {
    var userId = maxUserId++;
    var userName = "User " + userId;

    var user = {id: userId, name: userName, status: "ready"};
    Object.defineProperty(user, "socket", {
            value : socket,
            enumerable : false}
    );

    users[userId] = user;

    socket.emit("info", {text: "You have connected"});
    socket.emit("user-list", { users: users });
    socket.broadcast.emit("user-joined", user);
    return userId;
}

/**
 * Rozpoczyna rozgrywkę pomiędzy dwoma graczami, zaczyna nową sesję gry
 * aktualizuje stan lobby i oczekuje na sygnał końca gry
 * @param initiator gracz, który wyzwał innego gracza na pojedynek
 * @param target gracz wyzwany na pojedynek
 * @param initiatorRespond funkcja odpowiedzi
 */
function startGame(initiator, target, initiatorRespond) {
    initiator.status = target.status = "playing";

    var initGame = {
        player1: initiator,
        player2: target
    };

    initiatorRespond(initGame);
    target.socket.emit('challenged', initGame);

    var gameId = maxGameId++;

    io.sockets.emit("user-playing", initiator);
    io.sockets.emit("user-playing", target);

    new GameSession(gameId, target, initiator, function() {
        initiator.status = target.status = "ready";
        io.sockets.emit("user-ready", initiator);
        io.sockets.emit("user-ready", target);
    });
}
