/**
 * Menedżer obrazków jest odpowiedzialny za wczytywanie wielu obrazków i informowanie o postępie ładowania,  
 * błędach i końcu kolejki pobierania.
 */
function ImageManager(placeholderDataUri) {
	this._imageQueue = [];
	this._images = {};
	if (placeholderDataUri) {
		this._placeholder = new Image();
		this._placeholder.src = placeholderDataUri;
	}
}

_p = ImageManager.prototype;

/**
 * Dodaje obrazek do kolejki (pobieranie nie zostanie rozpoczęte do wywołania metody loadImages)
 * @param key - alias obrazka (używany w metodzie getImage)
 * @param path - ścieżka do pobrania
 */
_p._addImage = function(key, path) {
	this._imageQueue.push({
		key: key,
		path: path
	});
};

/**
 * Rozpoczyna proces pobierania i wysyła informacje o zdarzeniach
 * @param onProgress funkcja nasłuchująca o pięciu parametrach:
 * 		loaded: liczba wczytanych obrazków (prawidłowo lub nie)
 * 		total: całkowita liczba obrazków
 * 		key: alias wczytanego obrazka
 * 		path: ścieżka ostatnio wczytanego obrazka
 * 		success: true, jeśli ładowanie przebiegło prawidłowo
 * onProgress jest wywoływane po wczytaniu każdego obrazka
 */
_p.load = function(images, onDone, onProgress) {
    for (var im in images) {
        this._addImage(im, images[im]);
    }

    var noop = function() {};
    onDone = onDone || noop;
    onProgress = onProgress || noop;

	var queue = this._imageQueue;
	this._imageQueue = [];

	if (queue.length == 0) {
		onProgress(0, 0, null, null, true);
		return;
	}

	var itemCounter = {
		loaded: 0,
		total: queue.length
	};

	for (var i = 0; i < queue.length; i++) {
		this._loadItem(queue[i], itemCounter, onDone, onProgress);
	}
};

_p._loadItem = function(queueItem, itemCounter, onDone, onProgress) {
	var self = this;
	var img = new Image();
	img.onload = function() {
		self._images[queueItem.key] = img;
		self._onItemLoaded(queueItem, itemCounter, onDone, onProgress, true);
	};

	img.onerror = function() {
		self._images[queueItem.key] = self._placeholder ? self._placeholder : null;
		self._onItemLoaded(queueItem, itemCounter, onDone, onProgress, false);
	};
	img.src = queueItem.path;
};

_p._onItemLoaded = function(queueItem, itemCounter, onDone, onProgress, success) {
	itemCounter.loaded++;
	onProgress(itemCounter.loaded, itemCounter.total, queueItem.key, queueItem.path, success);
    if (itemCounter.loaded == itemCounter.total) {
        onDone();
    }
};

/**
 * Zwraca wczytany obrazek zgodnie z kluczem
 * @param alias obrazka
 */
_p.get = function(key) {
	return this._images[key];
};