/**
 * Podstawowa klasa do obsługi animacji
 * @param duration - czas trwania jednej iteracji animacji, w milisekundach
 * Np., animacja 6-ramkowa, dla której każda ramka trwa 50 milisekund, ma czas łączny równy 300 ms.
 *
 * Konstruktor przyjmuje jedynie konieczne parametry: wszystkie inne ustawienia przekazuje się przy użyciu setterów:
 * setRepeatBehavior and setRepeatCount (p. dokumentacja funkcji w dalszej części pliku)
 *
 * Klasa Animator nie obsługuje odmierzania czasu samodzielnie. Powinna ona być kontrolowana przy użyciu zewnętrznego timera.
 * Zewnętrzny timer wywołuje metodę update(), przekazując jako parametr lizcbę milisekund, jaka upłynęła od ostatniego wywołania tej funkcji.
 */
function Animator(duration) {
	this._duration = duration;

	// liczba pętli do wykonania
	this._repeatCount = 1;

	// ułamek przyspieszenia
	this._acceleration = 0;

	// ułamek opóźnienia
	this._deceleration = 0;

	// liczba wykonanych pętli
	this._loopsDone = 0;

	// Domyślne zachowanie animacji - powtarzanie
	this._repeatBehavior = Animator.RepeatBehavior.LOOP;

	// Czas, jaki upłynął od początku pętli
	this._timeSinceLoopStart = 0;

	// Ta flaga jest używana do oznaczenia startu animacji.
	this._started = false;
	this._running = false;

	// Flaga kontrolująca odtwarzanie w przeciwnym kierunku.
	this._reverseLoop = false;
}

_p = Animator.prototype;

/**
 * Jedna z wartości parametru dla funkcji setLoopCount
 */
Animator.INFINITE = -1;

/**
 * RepeatBehavior określa zachowanie klasy Animatora po zakończeniu pętli.
 * W przypadku wartości Animator.RepeatBehavior.LOOP, następna iteracja przebiegnie od 0 do 1.
 * W przypadku wartości Animator.RepeatBehavior.REVERSE nieparzyste pętle będą wykonywane odwrotnie
 * od 1 do 0. To ustawienie ma sens tylko, gdy liczba pętli jest większa od 1.
 */
Animator.RepeatBehavior = {
	LOOP: 1,
	REVERSE: 2
};

/**
 * Uruchamia klasę Animator. Po uruchomieniu nie można zmieniać wartości parametrów,
 * aż do wywołania metody stop().
 */
_p.start = function() {
	this._started = true;
	this._running = true;
};

/**
 * Sprawdza, czy klasa Animator jest uruchomiona.
 */
_p.isRunning = function() {
	return this._running;
};

/**
 * Zatrzymuje klasę Animator i przywraca stan początkowy. Funkcja może być używana wiele razy.
 */
_p.stop = function() {
	this._loopsDone = 0;
	this._timeSinceLoopStart = 0;
	this._running = false;
	this._started = false;
};

/**
 * Wstrzymuje klasę Animator. Animator ignoruje zmiany stanu, "zamrażając" stan animacji. 
 * Animacja może być przywrócona do działania od momentu zatrzymania.
 */
_p.pause = function() {
	this._running = false;
};

/**
 * Zwraca czas trwania jednej iteracji pętli animacji.
 */
_p.getDuration = function() {
	return this._duration;
};

/**
 * Ustawia czas trwania jednej iteracji pętli animacji. Wartość musi być większa lub równa 1.
 */
_p.setDuration = function(duration) {
	this._throwIfStarted();
	if (duration < 1) {
		throw "Czas trwania nie może być < 1";
	}
	this._duration = duration;
};

/**
 * Zwraca liczbę powtórzeń animacji - domyślnie 1.
 */
_p.getRepeatCount = function() {
	return this._repeatCount;
};

/**
 * Ustawia liczbę pętli animacji - domyślnie 1. Do poprawnych wartości należą liczby naturalne, a także Animator.INFINITE
 * w przypadku zapętlenia bez końca.
 */
_p.setRepeatCount = function(repeatCount) {
	this._throwIfStarted();

	if (repeatCount < 1 && repeatCount != Animator.INFINITE)
		throw "Liczba powtórzeń musi być większa od 0 lub INFINITE.";
	this._repeatCount = repeatCount;
};

/**
 * Zwraca sposób powtarzania animacji. 
 * Możliwe wartości to Animator.RepeatBehavior.LOOP i Animator.RepeatBehavior.REPATE. 
 * Więcej na ten temat w dokumentacji typu Animator.RepeatBehavior.
 */
_p.getRepeatBehavior = function() {
	return this._repeatBehavior;
};

/**
 * Ustawia sposób powtarzania - domyślna wartość to Animator.RepeatBehavior.LOOP
 * @param behavior nowy sposób powtarzania
 */
_p.setRepeatBehavior = function(behavior) {
	this._throwIfStarted();
	if (behavior != Animator.RepeatBehavior.LOOP && behavior != Animator.RepeatBehavior.REVERSE) {
		throw "Sposób powtarzania powinien przyjąć wartość RepeatBehavior.LOOP lub RepeatBehavior.REVERSE";
	}
	this._repeatBehavior = behavior;
};

/**
 * Zwraca aktualną wartość przyspieszenia (domyślnie 0).
 */
_p.getAcceleration = function() {
	return this._acceleration;
};

/**
 * Ustawia wartość przyspieszenia. Wartość musi być z przedziału (0, 1-deceleration).
 * @param acceleration nowe przyspieszenie
 */
_p.setAcceleration = function(acceleration) {
	this._throwIfStarted();
	if (acceleration < 0 || acceleration > 1 || acceleration > (1 - this._deceleration)) {
		throw "Przyspieszenie musi mieć wartość z przedziału (0, 1) i nie może być większe, niż (1 - deceleration)";
	}

	this._acceleration = acceleration;
};

/**
 * Zwraca aktualną wartość opóźnienia, domyślnie 0.
 */
_p.getDeceleration = function() {
	return this._deceleration;
};


/**
 * Ustawia wartość opóźnienia. Wartość musi być z przedziału (0, 1-deceleration).
 * @param acceleration nowe opóźnienie
 */
_p.setDeceleration = function(deceleration) {
	this._throwIfStarted();
	if (deceleration < 0 || deceleration > 1 || deceleration > (1 - this._acceleration)) {
		throw "Opóźnienie musi mieć wartość z przedziału (0, 1) i nie może być większe, niż (1 - acceleration)";
	}

	this._deceleration = deceleration;
};


/**
 * Domyślna implementacja preprocesora bierze pod uwagę tylko przyspieszenie i opóźnienie.
 * Bardziej zaawansowana implementacja może korzystać z funkcji
 * przejścia lub innych skomplikowanych mechanizmów.
 */
_p._timingEventPreprocessor = function(fraction) {
	return this._accelerationDecelerationPreprocessor(fraction);
};

/**
 * Oblicza aktualny ułamek, biorąc pod uwagę przyspieszenie i opóźnienie.
 * Zerknij do specyfikacji SMIL 2.0, aby zrozumieć zasady działania. Nie musisz analizować tego kodu,
 * jeśli nie jesteś naprawdę zainteresowany
 */
_p._accelerationDecelerationPreprocessor = function(fraction) {
	if (this._acceleration || this._deceleration) {
		var runRate = 1/(1 - this._acceleration/2 - this._deceleration/2);
		if (fraction < this._acceleration) {
			fraction *= runRate * (fraction / this._acceleration) / 2;
		} else if (fraction > (1 - this._deceleration)) {
			// czas spędzony w trakcie opóźnienia
			var tdec = fraction - (1 - this._deceleration);
			// stosunek tdec do całkowitego czasu opóźnienia
			var pdec  = tdec / this._deceleration;
			fraction = runRate * (1 - ( this._acceleration / 2) -
					this._deceleration + tdec * (2 - pdec) / 2);
		} else {
			fraction = runRate * (fraction - (this._acceleration / 2));
		}
		// ogranicz ułamek do przedziału [0,1], ponieważ powyższe obliczenia mogą
		// spowodować powstanie błędów zaokrąglenia.
		if (fraction < 0) {
			fraction = 0;
		} else if (fraction > 1) {
			fraction = 1;
		}
	}

	return fraction;
};

/**
 * Funkcja powinna być wywoływana przez zewnętrzny timer w celu aktualizacji stanu klasy Animator.
 * @param deltaTime - czas, jaki upłynął od ostatniego wywołania tej metody. 0 stanowi prawidłową wartość.
 */
_p.update = function(deltaTime) {

	// Zwróci undefined
	if (!this._started) {
		return;
	}

	// Jeśli klasa Animator została wstrzymana, przekazujemy 0 jako deltaTime - nic się nie zmieniło.
	if (!this._running) {
		deltaTime = 0;
	}

	this._timeSinceLoopStart += deltaTime;

	// Jeśli przekroczyliśmy czas trwania jednej iteracji, musimy odpowiednio zareagować:
	// dostosować kierunek animacji, obsłużyć haki, itd.
	if (this._timeSinceLoopStart >= this._duration) {

		// Na wszelki wypadek, jeśli pominęliśmy więcej, niż jedną iterację, musisz określić liczbę pominiętych pętli.
		var loopsSkipped = Math.floor(this._timeSinceLoopStart/this._duration);
		this._timeSinceLoopStart %= this._duration;

		// ogranicz uwzględniając liczbę pętli pominiętych. Nawet jeśli pominęliśmy pięć pętli,
		// a do końca zostały trzy, nie chcemy wywoływać dodatkowych funkcji nasłuchujących
		if (this._repeatCount != Animator.INFINITE && loopsSkipped > this._repeatCount - this._loopsDone) {
			loopsSkipped = this._repeatCount - this._loopsDone;
		}

		// wywołaj haki dla pominiętych iteracji
		for (var i = 1; i <= loopsSkipped; i++) {
			this._loopsDone++;
			this._reverseLoop = this._repeatBehavior == Animator.RepeatBehavior.REVERSE && this._loopsDone % 2 == 1;
			this._onLoopEnd(this._loopsDone);
		}

		// sprawdź, czy osiągnęliśmy koniec animacji
		if (this._repeatCount != Animator.INFINITE && this._loopsDone == this._repeatCount) {
			this._onAnimationEnd();
			this.stop();
			return;
		}
	}

	// Jeśli pętla jest wykonywana wstecz - odwróć ułamek
	var fraction = this._timeSinceLoopStart/this._duration;
	if (this._reverseLoop)
		fraction = 1 - fraction;

	// Wykonaj preprocessing (funkcje przejścia, itd.)
	fraction = this._timingEventPreprocessor(fraction);

	// Wykonaj aktualizację
	this._onUpdate(fraction, this._loopsDone);
	return fraction;
};

/**
 * Funkcja haka. Przesłoń w podklasie, aby dostarczyć własną implementację zmiany stanu.
 * Możesz np. zmienić aktywną ramkę, dostosować kolor lub kształt, itd.
 *
 * @param fraction aktualny ułamek animacji (od 0 do 1)
 * @param loopsDone liczba wykonanych pętli
 */
_p._onUpdate = function(fraction, loopsDone) {

};

/**
 * Funkcja haka - wywoływana w momencie zakończenia pętli. Powinna być przesłonięta w podklasach,
 * jeśli potrzebna jest dodatkowa funkcjonalność
 * @param loopsDone Liczba pętli od początku animacji. Uwzględnia ostatnią pętlę.
 */
_p._onLoopEnd = function(loopsDone) {

};

/**
 * Funkcja haka - wywoływana w momencie zakończenia animacji, do przesłonięcia w podklasach.
 */
_p._onAnimationEnd = function() {

};

/**
 * Funkcja wywoływana w celu sprawdzenia, czy klasa Animator nie jest uruchomiona.
 * Zmiana ustawień po uruchomieniu klasy Animatora nie jest możliwa.
 */
_p._throwIfStarted = function() {
	if (this._started)
		throw "Nie można zmieniać właściwości klasy Animatora po jego uruchomieniu";
};
