#![warn(rust_2018_idioms)]
#![allow(elided_lifetimes_in_paths)]

/// Kolejka LIFO operująca na znakach
pub struct Queue {
    older: Vec<char>,   // Starsze elementy, najstarszy ostatni
    younger: Vec<char>  // Młodsze elementy, najmłodszy ostatni
}

impl Queue {
    /// Wstaw znak na końcu kolejki
    pub fn push(&mut self, c: char) {
        self.younger.push(c);
    }

    /// Pobiera znak znajdujący się na początku kolejki. Zwraca `Some(c)`,
    /// jeśli jest znak do pobrania, `None`, gdy kolejka jest pusta
    pub fn pop(&mut self) -> Option<char> {
        if self.older.is_empty() {
            if self.younger.is_empty() {
                return None;
            }

            // Przenieś elementy nowsze do starszych,
            // ustaw je w określonej kolejności
            use std::mem::swap;
            swap(&mut self.older, &mut self.younger);
            self.older.reverse();
        }

        // Teraz starsze na pewno nie są puste. Metoda pop
        // już zwróciła Option
        self.older.pop()
    }
}

#[test]
fn test_push_pop() {
    let mut q = Queue { older: Vec::new(), younger: Vec::new() };

    q.push('0');
    q.push('1');
    assert_eq!(q.pop(), Some('0'));

    q.push('∞');
    assert_eq!(q.pop(), Some('1'));
    assert_eq!(q.pop(), Some('∞'));
    assert_eq!(q.pop(), None);

    (&mut q).push('0');
    (&mut q).push('1');
    assert_eq!(q.pop(), Some('0'));
    assert_eq!(q.pop(), Some('1'));
}

impl Queue {
    pub fn is_empty(&self) -> bool {
        self.older.is_empty() && self.younger.is_empty()
    }
}

#[test]
fn test_is_empty() {
    let mut q = Queue { older: Vec::new(), younger: Vec::new() };

    assert!(q.is_empty());
    q.push('☉');
    assert!(!q.is_empty());
    q.pop();
    assert!(q.is_empty());
}

impl Queue {
    pub fn split(self) -> (Vec<char>, Vec<char>) {
        (self.older, self.younger)
    }
}

#[test]
fn test_split() {
    let mut q = Queue { older: Vec::new(), younger: Vec::new() };

    q.push('P');
    q.push('D');
    assert_eq!(q.pop(), Some('P'));
    q.push('X');

    let (older, younger) = q.split();
    // q jest niezainicjalizowane
    assert_eq!(older, vec!['D']);
    assert_eq!(younger, vec!['X']);
}

impl Queue {
    pub fn new() -> Queue {
        Queue { older: Vec::new(), younger: Vec::new() }
    }
}

#[test]
fn test_new() {
    let mut q = Queue::new();

    q.push('*');
    q.push('1');
    assert_eq!(q.pop(), Some('*'));

    q.push('∞');
    assert_eq!(q.pop(), Some('1'));
    assert_eq!(q.pop(), Some('∞'));
    assert_eq!(q.pop(), None);
}
