#![warn(rust_2018_idioms)]
#![allow(elided_lifetimes_in_paths)]
#![allow(dead_code)]

mod gap {
    use std::ops::Range;

    /// A GapBuffer<T> to sekwencja elementów typu `T`, które mogą być 
    /// wstawiane i usuwane w dowolnym miejscu i w stałym czasie. Także odwołania
    /// (indeksowanie) są wykonywane w stałym czasie. Jednak zmiana miejsca
    /// wstawiania i usuwania jest wykonywana w czasie proporcjonalnym do odległości
    /// na jaką został przesunięty punkt wstawiania.
    pub struct GapBuffer<T> {
        // Przechowalnia elementów. Ma ona wymaganą przez nas pojemność, ale jej
        // długość zawsze jest równa 0. GapBuffer umieszcza swoje elementy i odstęp w
        // w tej „nieużywanej” pojemności `Vec`.
        storage: Vec<T>,

        // Zakres niezainicjowanych elementów w środku `przechowalni`.
        // Elementy znajdujące się przed tym zakresem i po nim są zawsze inicjowane.
        gap: Range<usize>
    }

    impl<T> GapBuffer<T> {
        pub fn new() -> GapBuffer<T> {
            GapBuffer { storage: Vec::new(), gap: 0..0 }
        }

        /// Zwraca liczbę elementów, które ten moduł GapBuffer może przechowywać
        /// bez potrzeby realokacji.
        pub fn capacity(&self) -> usize {
            self.storage.capacity()
        }

        /// Zwraca liczbę elementów przechowywanych w danej chwili.
        pub fn len(&self) -> usize {
            self.capacity() - self.gap.len()
        }

        /// Zwraca bieżącą pozycję wstawiania.
        pub fn position(&self) -> usize {
            self.gap.start
        }

        /// Zwraca wskaźnik do `index`-owego elementu przechowalni
        /// bez względu na odstęp.
        ///
        /// Bezpieczeństwo: `index` musi być prawidłowym indeksem `self.storage`.
        unsafe fn space(&self, index: usize) -> *const T {
            self.storage.as_ptr().offset(index as isize)
        }

        /// Zwraca wskaźnik mutowalny do `index`-owego elementu przechowalni
        /// bez względu na odstęp.
        ///
        /// Bezpieczeństwo: `index` musi być prawidłowym indeksem `self.storage`.
        unsafe fn space_mut(&mut self, index: usize) -> *mut T {
            self.storage.as_mut_ptr().offset(index as isize)
        }

        /// Zwraca przesunięcie w buforze`index`-owego elementu biorąc pod uwagę
        /// odstęp. Funkcja ta nie sprawdza, czy indeks znajduje się w danym zakresie, ale
        /// nigdy nie zwraca indeksu znajdującego się wewnątrz odstępu.
        fn index_to_raw(&self, index: usize) -> usize {
            if index < self.gap.start {
                index
            } else {
                index + self.gap.len()
            }
        }

        /// Zwraca referencję do `index`'-owego elementu
        /// albo `None`, jeżeli `index` znajduje się poza granicami.
        pub fn get(&self, index: usize) -> Option<&T> {
            let raw = self.index_to_raw(index);
            if raw < self.capacity() {
                unsafe {
                    // Porównaliśmy `raw` z self.capacity()
                    // i index_to_raw pomija odstęp, więc to rozwiązanie jest bezpieczne.
                    Some(&*self.space(raw))
                }
            } else {
                None
            }
        }

        /// Wyznacza `pos` jako bieżącą pozycję wstawiania.
        /// Jeżeli `pos` znajduje się poza granicami, zostaje zgłoszony błąd paniki.
        pub fn set_position(&mut self, pos: usize) {
            if pos > self.len() {
                panic!("indeks {} znajduje się poza zakresem dla GapBuffer", pos);
            }

            unsafe {
                let gap = self.gap.clone();
                if pos > gap.start {
                    // `pos` znajduje się za odstępem. Przenosi odstęp w prawo poprzez
                    // przesunięcie elementów znajdujących się po odstępie przed niego.
                    let distance = pos - gap.start;
                    std::ptr::copy(self.space(gap.end),
                                   self.space_mut(gap.start),
                                   distance);
                } else if pos < gap.start {
                    // `pos` znajduje się przed odstępem. Przenosi odstęp w lewo poprzez
                    // przesunięcie elementów znajdujących się przed odstępem za niego.
                    let distance = gap.start - pos;
                    std::ptr::copy(self.space(pos),
                                   self.space_mut(gap.end - distance),
                                   distance);
                }

                self.gap = pos .. pos + gap.len();
            }
        }

        /// Umieszcza `elt` w bieżącej pozycji wstawiania i
        /// pozostawia za nią pozycję wstawiania.
        pub fn insert(&mut self, elt: T) {
            if self.gap.len() == 0 {
                self.enlarge_gap();
            }

            unsafe {
                let index = self.gap.start;
                std::ptr::write(self.space_mut(index), elt);
            }
            self.gap.start += 1;
        }

        /// Umieszcza elementy wygenerowane przez `iter` w bieżącej pozycji wstawiania i
        /// pozostawia za nimi pozycję wstawiania.
        pub fn insert_iter<I>(&mut self, iterable: I)
            where I: IntoIterator<Item=T>
        {
            for item in iterable {
                self.insert(item)
            }
        }

        /// Usuwa element tuż za pozycją wstawiania i zwraca ją lub
        /// zwraca `None`, jeżeli pozycja wstawiania 
        /// znajduje się na końcu GapBuffer.
        pub fn remove(&mut self) -> Option<T> {
            if self.gap.end == self.capacity() {
                return None;
            }

            let element = unsafe {
                std::ptr::read(self.space(self.gap.end))
            };
            self.gap.end += 1;
            Some(element)
        }

        /// Podwaja pojemność `self.storage`.
        fn enlarge_gap(&mut self) {
            let mut new_capacity = self.capacity() * 2;
            if new_capacity == 0 {
                // Istniejący wektor jest pusty.
                // Wybiera rozsądną pojemność początkową.
                new_capacity = 4;
            }

            // Nie mamy pojęcia, w jaki sposób Vec zmienia rozmiar "nieużywanej" pojemności.
            // Z tego powodu zostaje po prostu utworzony nowy wektor, do którego zostają
            // przeniesione elementy.
            let mut new = Vec::with_capacity(new_capacity);
            let after_gap = self.capacity() - self.gap.end;
            let new_gap = self.gap.start .. new.capacity() - after_gap;

            unsafe {
                // Przenosi elementy znajdujące się przed odstępem.
                std::ptr::copy_nonoverlapping(self.space(0),
                                              new.as_mut_ptr(),
                                              self.gap.start);

                // Przenosi elementy znajdujące się za odstępem.
                let new_gap_end = new.as_mut_ptr().offset(new_gap.end as isize);
                std::ptr::copy_nonoverlapping(self.space(self.gap.end),
                                              new_gap_end,
                                              after_gap);
            }

            // Zostaje tu zwolniony stary Vec, ale żaden element nie zostaje porzucony,
            // ponieważ długość wektora wynosi 0.
            self.storage = new;
            self.gap = new_gap;
        }
    }

    impl<T> Drop for GapBuffer<T> {
        fn drop(&mut self) {
            unsafe {
                for i in 0 .. self.gap.start {
                    std::ptr::drop_in_place(self.space_mut(i));
                }
                for i in self.gap.end .. self.capacity() {
                    std::ptr::drop_in_place(self.space_mut(i));
                }
            }
        }
    }

    pub struct Iter<'a, T> {
        buffer: &'a GapBuffer<T>,
        pos: usize
    }

    impl<'a, T> Iterator for Iter<'a, T> {
        type Item = &'a T;
        fn next(&mut self) -> Option<&'a T> {
            if self.pos >= self.buffer.len() {
                None
            } else {
                self.pos += 1;
                self.buffer.get(self.pos - 1)
            }
        }
    }

    impl<'a, T: 'a> IntoIterator for &'a GapBuffer<T> {
        type Item = &'a T;
        type IntoIter = Iter<'a, T>;
        fn into_iter(self) -> Iter<'a, T> {
            Iter { buffer: self, pos: 0 }
        }
    }

    impl GapBuffer<char> {
        pub fn get_string(&self) -> String {
            let mut text = String::new();
            text.extend(self);
            text
        }
    }

    use std::fmt;
    impl<T: fmt::Debug> fmt::Debug for GapBuffer<T> {
        fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
            let indices = (0..self.gap.start).chain(self.gap.end .. self.capacity());
            let elements = indices.map(|i| unsafe { &*self.space(i) });
            f.debug_list().entries(elements).finish()
        }
    }
}


mod gap_tests {
    #[test]
    fn test() {
        use super::gap::GapBuffer;

        let mut buf = GapBuffer::new();
        buf.insert_iter("Lord of the Rings".chars());
        buf.set_position(12);

        buf.insert_iter("Onion ".chars());

        assert_eq!(buf.get_string(), "Lord of the Onion Rings");
    }

    #[test]
    fn misc() {
        use super::gap::GapBuffer;

        let mut gb = GapBuffer::new();
        println!("{:?}", gb);
        gb.insert("foo".to_string());
        println!("{:?}", gb);
        gb.insert("bar".to_string());
        println!("{:?}", gb);
        gb.insert("baz".to_string());
        println!("{:?}", gb);
        gb.insert("qux".to_string());
        println!("{:?}", gb);
        gb.insert("quux".to_string());
        println!("{:?}", gb);

        gb.set_position(2);

        assert_eq!(gb.remove(), Some("baz".to_string()));
        println!("{:?}", gb);
        assert_eq!(gb.remove(), Some("qux".to_string()));
        println!("{:?}", gb);
        assert_eq!(gb.remove(), Some("quux".to_string()));
        println!("{:?}", gb);
        assert_eq!(gb.remove(), None);
        println!("{:?}", gb);

        gb.insert("quuux".to_string());
        println!("{:?}", gb);

        gb.set_position(0);
        assert_eq!(gb.remove(), Some("foo".to_string()));
        println!("{:?}", gb);
        assert_eq!(gb.remove(), Some("bar".to_string()));
        println!("{:?}", gb);
        assert_eq!(gb.remove(), Some("quuux".to_string()));
        println!("{:?}", gb);
        assert_eq!(gb.remove(), None);
        println!("{:?}", gb);
    }

    #[test]
    fn drop_elements() {
        use super::gap::GapBuffer;

        let mut gb = GapBuffer::new();
        gb.insert("foo".to_string());
        gb.insert("bar".to_string());

        gb.set_position(1);
    }
}
