mod my_ascii {
    /// Łańcuch znaków kodowany w formacie ASCII.
    #[derive(Debug, Eq, PartialEq)]
    pub struct Ascii(
        // Tutaj musi być przechowywany wyłącznie poprawny tekst ASCII:
        // bajty od `0` do `0x7f`.
        Vec<u8>
    );

    impl Ascii {
        /// Tworzy `Ascii` z tekstu ASCII umieszczonego w `bytes`. Zwraca
        /// błąd `NotAsciiError`, jeżeli w `bytes` występuje jakikolwiek znak
        /// niekodowany w formacie ASCII.
        pub fn from_bytes(bytes: Vec<u8>) -> Result<Ascii, NotAsciiError> {
            if bytes.iter().any(|&byte| !byte.is_ascii()) {
                return Err(NotAsciiError(bytes));
            }
            Ok(Ascii(bytes))
        }
    }

    // W przypadku nieudanej konwersji zostaje zwrócony wektor, którego nie  można było
    // przekonwertować. Należałoby zaimplementować `std::error::Error`; pominięto dla
    // zachowania zwięzłości.
    #[derive(Debug, Eq, PartialEq)]
    pub struct NotAsciiError(pub Vec<u8>);

    // Bezpieczna, skuteczna konwersja zaimplementowana za pomocą kodu niebezpiecznego.
    impl From<Ascii> for String {
        fn from(ascii: Ascii) -> String {
            // Jeżeli moduł ten jest pozbawiony błędów, okazuje się bezpieczny, gdyż
            // poprawny tekst ASCII stanowi jednocześnie poprawny kod UTF-8.
            unsafe { String::from_utf8_unchecked(ascii.0) }
        }
    }

    // Fragment ten należy umieścić wewnątrz modułu `my_ascii`.
    impl Ascii {
        /// Konstruuje wartość `Ascii` z elementu `bytes` bez sprawdzania poprawności
        /// zamieszczonego w nim tekstu ASCII.
        ///
        /// Konstruktor ten jest nieomylny i bezpośrednio zwraca wartość `Ascii`,
        /// a nie poprzez `Result<Ascii, NotAsciiError>`, jak to ma miejsce
        /// w przypadku konstruktora `from_bytes`.
        ///
        /// # Bezpieczeństwo
        ///
        /// Kod wywołujący musi zagwarantować, że `bytes` zawiera wyłącznie znaki ASCII:
        /// bajty o wartości nie większej niż 0x7f. W przeciwnym wypadku skutki stają się
        /// niezdefiniowane.
        pub unsafe fn from_bytes_unchecked(bytes: Vec<u8>) -> Ascii {
            Ascii(bytes)
        }
    }
}

#[test]
fn good_ascii() {
    use my_ascii::Ascii;

    let bytes: Vec<u8> = b"ASCII a zostanie wam dane".to_vec();

    // Wywołanie to nie wymaga alokacji ani kopii tekstu, jedynie przeglądania.
    let ascii: Ascii = Ascii::from_bytes(bytes)
        .unwrap(); // Wiemy, że te wybrane bajty są prawidłowe.

    // This call is zero-cost: no allocation, copies, or scans.
    let string = String::from(ascii);

    assert_eq!(string, "ASCII a zostanie wam dane");
}


#[test]
fn bad_ascii() {
    use my_ascii::Ascii;

    // Przyjmijmy, że poniższy wektor stanowi wynik jakiegoś skomplikowanego procesu,
    // który miał wygenerować tekst w formacie ASCII. Coś poszło nie tak!
    let bytes = vec![0xf7, 0xbf, 0xbf, 0xbf];

    let ascii = unsafe {
        // Ten kontrakt funkcji niebezpiecznej został naruszony z powodu
        // przechowywania bajtów niekodowanych w formacie ASCII przez `bytes`.
        Ascii::from_bytes_unchecked(bytes)
    };

    let bogus: String = ascii.into();

    // `bogus` przechowuje teraz nieprawidłowe kodowanie UTF-8. 
    // Analiza składniowa pierwszego znaku generuje `char`
    // zawierający niewłaściwe kodowanie Unicode.
    assert_eq!(bogus.chars().next().unwrap() as u32, 0x1fffff);
}
