//! Struktura biologiczna wyższego poziomu.
//!
//! Zawsze symulujemy próbki wszystkich chemicznych interakcji na poziomie
//! komórkowym, jednak symulowanie wszystkiego w taki sposób jest po prostu zbyt 
//! kosztowne obliczeniowo. Dlatego przechowujemy struktury danych wyższego poziomu,
//! reprezentujące korzenie, liście, itd. paproci. Symulując zjawiska fizyczne,
//! takie jak: światło, ruchy powietrza, grawitację, itd.) zawsze używamy tych
//! struktur danych, jako swoistych skrótów do milionów komórek, z których
//! zazwyczaj się one składają. W bardziej chorobliwym tonie, struktury te 
//! utrzymują się, gdy rzeczy umierają, więc martwe listki mają wagę, 
//! rzucają cień i tak dalej.

// in plant_structures/mod.rs
pub mod roots;
pub mod stems;
pub mod leaves;

pub use self::leaves::Leaf;
pub use self::roots::Root;

use self::roots::RootSet;
use self::stems::StemSet;

pub enum FernType {
    Fiddlehead
}

pub struct Fern {
    pub roots: RootSet,
    pub stems: StemSet
}

impl Fern {
    pub fn new(_type: FernType) -> Fern {
        Fern {
            roots: vec![],
            stems: vec![stems::Stem { furled: true }]
        }
    }

    pub fn is_furled(&self) -> bool { !self.is_fully_unfurled() }

    pub fn is_fully_unfurled(&self) -> bool {
        self.stems.iter().all(|s| !s.furled)
    }
}

/// Tworzy i zwraca [`VascularPath`] reprezentującą ścieżkę składników
/// pokarmowych z podanego korzenia, [`Root`][r], do podanego liścia, [`Leaf`](leaves::Leaf).
///
/// [r]: roots::Root
pub fn trace_path(leaf: &leaves::Leaf, root: &roots::Root) -> VascularPath {
    VascularPath { from: leaf.x, to: root.x }
}

#[doc(alias = "route")]
pub struct VascularPath {
    pub from: bool,
    pub to: bool,
}
