#![warn(rust_2018_idioms)]
#![allow(elided_lifetimes_in_paths)]

use num::Complex;
use rayon::prelude::*;

/// WyznaczamySprawdzamy, czy `c` należy do zbioru Mandelbrota, ustalając `limit`
/// iteracji.
///
/// Jeśli `c` nie należy do zbioru, zwracamy `Some(i)`, gdzie `i` jest liczbą
/// iteracji do momentu, gdy `c` opuściło okrąg o promieniu 2 razy większym
/// od punktu początkowego. Jeśli `c` jest kandydatem na element zbioru
/// (osiągnęliśmy limit iteracji bez wyniku negatywnego), zwracamy `None`.
fn escape_time(c: Complex<f64>, limit: usize) -> Option<usize> {
    let mut z = Complex { re: 0.0, im: 0.0 };
    for i in 0..limit {
        if z.norm_sqr() > 4.0 {
            return Some(i);
        }
        z = z * z + c;
    }

    None
}

use std::str::FromStr;

/// Parsowany string `s` to para współrzędnych, np. `"400x600"` lub `"1.0,0.5"`.
///
/// Ogólnie: `s` jest w postaci <lewa><separator><prawa>, gdzie <separator> jest 
/// znakiem przekazanym jako wartość parametru `separator`, a <lewa> i <prawa>
/// to wartości, które mogą być parsowane za pomocą `T::from_str`. Parametr
/// `separator` musi być znakiem ASCII.
///
/// Jeśli `s` ma prawidłową postać, zwracamy `Some<(x, y)>`. Jeśli nie,
/// zwracamy `None`.
fn parse_pair<T: FromStr>(s: &str, separator: char) -> Option<(T, T)> {
    match s.find(separator) {
        None => None,
        Some(index) => {
            match (T::from_str(&s[..index]), T::from_str(&s[index + 1..])) {
                (Ok(l), Ok(r)) => Some((l, r)),
                _ => None
            }
        }
    }
}

#[test]
fn test_parse_pair() {
    assert_eq!(parse_pair::<i32>("",        ','), None);
    assert_eq!(parse_pair::<i32>("10,",     ','), None);
    assert_eq!(parse_pair::<i32>(",10",     ','), None);
    assert_eq!(parse_pair::<i32>("10,20",   ','), Some((10, 20)));
    assert_eq!(parse_pair::<i32>("10,20xy", ','), None);
    assert_eq!(parse_pair::<f64>("0.5x",    'x'), None);
    assert_eq!(parse_pair::<f64>("0.5x1.5", 'x'), Some((0.5, 1.5)));
}

/// Zamienia dwie liczby zmiennoprzecinkowe oddzielone separatorem
/// na liczbę zespoloną.
fn parse_complex(s: &str) -> Option<Complex<f64>> {
    match parse_pair(s, ',') {
        Some((re, im)) => Some(Complex { re, im }),
        None => None
    }
}

#[test]
fn test_parse_complex() {
    assert_eq!(parse_complex("1.25,-0.0625"),
               Some(Complex { re: 1.25, im: -0.0625 }));
    assert_eq!(parse_complex(",-0.0625"), None);
}

/// Na podstawie współrzędnych graficznych punktu (numer wiersza i kolumny)
/// zwraca punkt reprezentowany przez liczbę zespoloną.
///
/// `bounds` to para określająca wysokość i szerokość obrazu w pikselach,
/// `pixel` to para (kolumna, wiersz) wskazująca konkretny piksel obrazu,
/// parametry `upper_left` i `lower_right` to punkty określające lewy górny
/// i prawy dolny wierzchołek płaszczyzny liczb zespolonych.
fn pixel_to_point(bounds: (usize, usize),
                  pixel: (usize, usize),
                  upper_left: Complex<f64>,
                  lower_right: Complex<f64>)
    -> Complex<f64>
{
    let (width, height) = (lower_right.re - upper_left.re,
                           upper_left.im - lower_right.im);
    Complex {
        re: upper_left.re + pixel.0 as f64 * width  / bounds.0 as f64,
        im: upper_left.im - pixel.1 as f64 * height / bounds.1 as f64
        // Skąd tutaj odejmowanie? pixel.1 zwiększa wartość, idąc w dół obrazu,
        // podczas gdy jego zespolony odpowiednik, zwiększając wartość, idzie w górę
    }
}

#[test]
fn test_pixel_to_point() {
    assert_eq!(pixel_to_point((100, 200), (25, 175),
                              Complex { re: -1.0, im:  1.0 },
                              Complex { re:  1.0, im: -1.0 }),
               Complex { re: -0.5, im: -0.75 });
}

/// Rysowanie zbioru Mandelbrota na płaszczyźnie o podanych rozmiarach
///
/// `bounds` to para określająca wysokość i szerokość obrazu w pikselach,
/// `pixel` to bufor wszystkich pikseli obrazu, pojedyncza wartość bufora
/// to kolor (poziom szarości) piksela podany jako bajt (u8),
/// `upper_left` i `lower_right` to punkty wskazujące na wierzchołki obrazu
/// wyrażone jako liczby zespolone układu mapowanego na piksele obrazu
fn render(pixels: &mut [u8],
          bounds: (usize, usize),
          upper_left: Complex<f64>,
          lower_right: Complex<f64>)
{
    assert!(pixels.len() == bounds.0 * bounds.1);

    for row in 0..bounds.1 {
        for column in 0..bounds.0 {
            let point = pixel_to_point(bounds, (column, row),
                                       upper_left, lower_right);
            pixels[row * bounds.0 + column] =
                match escape_time(point, 255) {
                    None => 0,
                    Some(count) => 255 - count as u8
                };
        }
    }
}

use image::ColorType;
use image::png::PNGEncoder;
use std::fs::File;

/// Zapis zawartości bufora `pixels` w postaci obrazu o rozmiarach `bounds`
/// do pliku o nazwie `filename`
fn write_image(filename: &str, pixels: &[u8], bounds: (usize, usize))
    -> Result<(), std::io::Error>
{
    let output = File::create(filename)?;

    let encoder = PNGEncoder::new(output);
    encoder.encode(&pixels,
                   bounds.0 as u32, bounds.1 as u32,
                   ColorType::Gray(8))?;

    Ok(())
}

use std::env;

fn main() {
    let args: Vec<String> = env::args().collect();

    if args.len() != 5 {
        eprintln!("Sposób użycia: {} FILE PIXELS UPPERLEFT LOWERRIGHT",
                  args[0]);
        eprintln!("przykład: {} mandel.png 1000x750 -1.20,0.35 -1,0.20",
                  args[0]);
        std::process::exit(1);
    }

    let bounds = parse_pair(&args[2], 'x')
        .expect("Błąd parsowania rozmiaru obrazu");
    let upper_left = parse_complex(&args[3])
        .expect("Błąd parsowania lewego górnego wierzchołka");
    let lower_right = parse_complex(&args[4])
        .expect("Błąd parsowania prawego dolnego wierzchołka");

    let mut pixels = vec![0; bounds.0 * bounds.1];

    // Zakres podziału `pixels` na poziome paski.
    {
        let bands: Vec<(usize, &mut [u8])> = pixels
            .chunks_mut(bounds.0)
            .enumerate()
            .collect();

        bands.into_par_iter()
            .for_each(|(i, band)| {
                let top = i;
                let band_bounds = (bounds.0, 1);
                let band_upper_left = pixel_to_point(bounds, (0, top),
                                                     upper_left, lower_right);
                let band_lower_right = pixel_to_point(bounds, (bounds.0, top + 1),
                                                      upper_left, lower_right);
                render(band, band_bounds, band_upper_left, band_lower_right);
            });
    }

    write_image(&args[1], &pixels, bounds)
        .expect("Błąd zapisu pliku PNG");
}
