//! Indeksy przechowywane w pamięci.
//!
//! Pierwszym krokiem tworzenia indeksu jest zindeksowanie dokumentu w pamięci.
//! Do tego celu można użyć indeksu `InMemoryIndex`, który może wykorzystać
//! całą pamięć dostępną na komputerze

use std::collections::HashMap;
use byteorder::{LittleEndian, WriteBytesExt};

/// Break a string into words.
fn tokenize(text: &str) -> Vec<&str> {
    text.split(|ch: char| !ch.is_alphanumeric())
        .filter(|word| !word.is_empty())
        .collect()
}

/// Indeks przechowywany w pamięci.
///
/// Oczywiście, rzeczywisty indeks dużego korpusu dokumentów nie zmieści się 
/// w pamięci. Jednak pomijając ograniczenia pamięciowe, ten indeks jest wszystkim
/// czego potrzeba by odpowiadać na proste zapytania. Do zapisu tych indeksów 
/// w plikach na dysku, scalania ich ze sobą i tworzenia dużych indeksów, można 
/// użyć modułów `read`, `write` oraz `merge`
pub struct InMemoryIndex {
    /// Całkowita liczba słów w indeksowanych dokumentach
    pub word_count: usize,

    /// Lista trafień dla wszystkich terminów występujących w indeksie 
    /// (czyli nazwa dokumentu i określenie miejsc, w których dany termin 
    /// występuje).
    ///
    /// Może się zdarzyć, że indeks będzie "sortowany na podstawie id dokumentu",
    /// czyli dla każdego `Vec<Hit>` w tej mapie, wszystkie `Hit` będą zawierać
    /// unikalne identyfikatory dokumentów (pierwszą wartość u32) oraz że elementy
    /// `Hit` będą zapisane w rosnącej kolejności identyfikatorów dokumentów.
    /// To przydatne dla niektórych algorytmów, które można wykonywać na indeksach,
    /// więc, o ile to możliwe, warto zachować tę cechę
    pub map: HashMap<String, Vec<Hit>>
}

/// `Hit` oznacza, że konkretny dokument zawiera okreśony termin, zawiera także
/// informację o liczbie wystąpień tego terminu oraz przesunięcia wskazujące 
/// miejsca jego wystąpień (podawane jako liczba słów od początku dokumentu do 
/// miejsca wystąpienia terminu).
///
/// Bufor zawiera wszystkie dane trafień w formie binarnej, zapisany w notacji
/// little-endian, najmniej znaczący bajt na początku. Pierwsza wartość u32 danych
/// to indeks dokumentu, pozostałe [u32] to przesunięcia
pub type Hit = Vec<u8>;

impl InMemoryIndex {
    /// Tworzymy nowy pusty indeks
    pub fn new() -> InMemoryIndex {
        InMemoryIndex {
            word_count: 0,
            map: HashMap::new()
        }
    }

    /// Indeksuje jeden dokument
    ///
    /// Wynikowy indeks zawiera dokładnie jeden `Hit` na termin
    pub fn from_single_document(document_id: usize, text: String) -> InMemoryIndex {
        let document_id = document_id as u32;
        let mut index = InMemoryIndex::new();

        let text = text.to_lowercase();
        let tokens = tokenize(&text);
        for (i, token) in tokens.iter().enumerate() {
            let hits =
                index.map
                .entry(token.to_string())
                .or_insert_with(|| {
                    let mut hits = Vec::with_capacity(4 + 4);
                    hits.write_u32::<LittleEndian>(document_id).unwrap();
                    vec![hits]
                });
            hits[0].write_u32::<LittleEndian>(i as u32).unwrap();
            index.word_count += 1;
        }

        if document_id % 100 == 0 {
            println!("zindeksowano dokument {}, {} bajtów, {} słów", document_id, text.len(), index.word_count);
        }

        index
    }

    /// Dodaje do tego indeksu wszystkie trafienia z `other`.
    ///
    /// Jeśli zarówno `*self` jak i `other` są posortowane według identyfikatora
    /// dokumentu, oraz wszystkie identyfikatory dokumentów w `other` są większe 
    /// od dowolnego identyfikator dokumentu w `*self`, to po scaleniu `*self`
    /// wciąż będzie posortowany według identyfikatorów dokumentów
    pub fn merge(&mut self, other: InMemoryIndex) {
        for (term, hits) in other.map {
            self.map.entry(term)
                .or_insert_with(|| vec![])
                .extend(hits)
        }
        self.word_count += other.word_count;
    }

    /// Zwraca prawdę jeśli indeks jest pusty
    pub fn is_empty(&self) -> bool {
        self.word_count == 0
    }

    /// Zwraca prawdę jeśli indeks jest indeks jest dostatecznie duży, by należało
    /// go zapisać w pliku na dysku zamiast dodawać c niego kolejne dane
    pub fn is_large(&self) -> bool {
        // Oczywiście, wszystko zależy ile pamięci jest zainstalowanych w komputerze
        const REASONABLE_SIZE: usize = 100_000_000;
        self.word_count > REASONABLE_SIZE
    }
}
