#ifndef SERVICE_H
#define SERVICE_H

// Biblioteka standardowa
#include <iostream>
#include <functional>

// Symulujemy strumie reaktywny za pomoc zakresw
#include <range/v3/view/filter.hpp>
#include <range/v3/view/transform.hpp>
#include <range/v3/to_container.hpp>

/**
 * Struktura, ktra zawiera warto dowolnego typu,
 * a take informacj kontekstow - wiadomo,
 * ktr powinien wysa serwer w odpowiedzi.
 */
template <typename MessageType>
struct with_expected_reply {
    MessageType value;
    std::string expected_reply;

    void reply(const std::string& message) const
    {
        REQUIRE(message == expected_reply);
    }
};

/**
 * Funkcja, ktra tworzy instancj typu with_expected_reply 
 * majc dan warto i gniazdo.
 */
template <typename MessageType>
auto make_with_expected_reply(MessageType&& value, const std::string& expected_reply)
{
    return with_expected_reply<MessageType>{
        std::forward<MessageType>(value), expected_reply};
}

 /**
 * Podnosi dowoln funkcj przeksztacajc typ T1 na T2
 * do funkcji przeksztacajcej typ with_expected_reply<T1> na with_expected_reply<T2>.
 */
template <typename F>
auto lift_with_expected_reply(F&& function)
{
    return [function = std::forward<F>(function)] (auto &&ws) {
        return make_with_expected_reply(std::invoke(function, ws.value), ws.expected_reply);
    };
}

 /**
 * Konwertuje funkcj przeksztacajc typ T1 na T2 na funkcj
 * przeksztacajc typ with_expected_reply<T1> na T2.
 * Nie jest to przydatne dla wszystkich monad, ale potrzebujemy tej funkcji,
 * aby mc uy transformacji filter dla strumieni, ktre wysyaj wartoci z oczekiwanymi odpowiedziami.
 */
template <typename F>
auto apply_with_expected_reply(F&& function)
{
    return [function = std::forward<F>(function)] (auto &&ws) {
        return std::invoke(function, std::forward<decltype(ws)>(ws).value);
    };
}

#endif
