#ifndef SERVICE_H
#define SERVICE_H

// Biblioteka standardowa
#include <iostream>
#include <functional>

// Boost ASIO
#include <boost/asio.hpp>

// Nie jest dobrym pomysem umieszczenie deklaracji using w pliku nagwkowym.
// ale w przypadku tego niewielkiego przykadu jako to przeyjemy.
using boost::asio::ip::tcp;


/**
 * Klasa obsugujca sesj.
 *
 * Odczytuje dane wysane przez klienta wiersz po wierszu,
 * a nastpnie wysya kady z wierszy w postaci oddzielnej wiadomoci.
 */
template <typename EmitFunction>
class session: public std::enable_shared_from_this<session<EmitFunction>> {
public:
    session(tcp::socket&& socket, EmitFunction emit)
        : m_socket(std::move(socket))
        , m_emit(emit)
    {
    }

    void start()
    {
        do_read();
    }

private:
    using shared_session = std::enable_shared_from_this<session<EmitFunction>>;

    void do_read()
    {
		// Uzyskiwanie wspdzielonego wskanika do tej instancji,
		// aby przechwyci go w wyraeniu lambda.
        auto self = shared_session::shared_from_this();
        boost::asio::async_read_until(
            m_socket, m_data, '\n',
            [this, self](const boost::system::error_code& error,
                         std::size_t size) {
                if (!error) {
					// Odczytywanie wiersza tekstu
					// i przekazywanie go tym klientom, ktrzy nas nasuchuj.
                    std::istream is(&m_data);
                    std::string line;
                    std::getline(is, line);
                    m_emit(std::move(line));

					// Zaplanowanie odczytywania kolejnego wiersza.
                    do_read();
                }
            });
    }

    tcp::socket m_socket;
    boost::asio::streambuf m_data;
    EmitFunction m_emit;
};

/**
 * Tworzy wspdzielony wskanik do sesji okrelanej przez gniazdo
 * oraz definiuje funkcj, ktra zostanie uyta przez sesj
 * do wysyania wiadomoci.
 */
template <typename Socket, typename EmitFunction>
auto make_shared_session(Socket&& socket, EmitFunction&& emit)
{
    return std::make_shared<session<EmitFunction>>(
            std::forward<Socket>(socket),
            std::forward<EmitFunction>(emit));
}

/**
 * Klasa service obsuguje poczenia od klientw
 * oraz wysya wiadomoci odebrane od nich.
 */
class service {
public:
    using value_type = std::string;

    explicit service(boost::asio::io_service& service,
                     unsigned short port = 42042);

    service(const service &other) = delete;
    service(service &&other) = default;

    template <typename EmitFunction>
    void on_message(EmitFunction emit)
    {
        m_emit = emit;
        do_accept();
    }

private:
    void do_accept();

    tcp::acceptor m_acceptor;
    tcp::socket m_socket;
    std::function<void(std::string&&)> m_emit;

    friend std::ostream& operator<< (std::ostream& out, const service& service)
    {
        return out << "obiekt service";
    }
};

#endif
