/******************************************************************************
 *  Kompilacja:  javac ST.java
 *  Wykonanie:    java ST
 *  Zależności: StdIn.java StdOut.java
 *  
 *  Implementacja sortowanej tablicy symboli z użyciem typu java.util.TreeMap.
 *  Duplikaty są niedozwolone.
 *
 *  % java ST
 *
 ******************************************************************************/

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.TreeMap;

/**
 *  Klasa {@code ST} reprezentuje uporządkowaną tablicę symboli generycznych par
 *  klucz-wartość.
 *  Obsługuje standardowe metody <em>put</em>, <em>get</em>, <em>contains</em>,
 *  <em>remove</em>, <em>size</em> i <em>is-empty</em>.
 *  Udostępnia też metody do znajdowania elementów w uporządkowanej tablicy: <em>minimum</em>,
 *  <em>maximum</em>, <em>floor</em> i <em>ceiling</em>.
 *  Ponadto udostępnia metodę <em>keys</em> do iteracyjnego przechodzenia po wszystkich kluczach.
 *  Tablica symboli to implementacja abstrakcji <em>tablic asocjacyjnych</em>:
 *  gdy wartość jest łączona z kluczem znajdującym się już w tablicy symboli,
 *  standardowo dawna wartość zostaje zastąpiona nową.
 *  Inaczej niż w klasie {@link java.util.Map} tu przyjmuje się, że
 *  wartości nie mogą być równe {@code null}. Ustawienie wartości
 *  klucza na {@code null} to odpowiednik usunięcia tego klucza
 *  z tablicy symboli.
 *  <p>
 *  W tej implementacji używane jest drzewo BST. To wymaga, by klucz
 *  implementował interfejs {@code Comparable} i wywoływał
 *  {@code compareTo()} do porównywania dwóch kluczy. Kod nie wywołuje
 *  {@code equals()} ani {@code hashCode()}.
 *  Operacje <em>put</em>, <em>contains</em>, <em>remove</em>, <em>minimum</em>,
 *  <em>maximum</em>, <em>ceiling</em> i <em>floor</em> w najgorszej sytuacji
 *  działają w czasie logarytmicznym.
 *  Operacje <em>size</em> i <em>is-empty</em> działają w stałym czasie.
 *  Konstrukcja obiektów zajmuje stały czas.
 *  <p>
 *  Dodatkową dokumentację znajdziesz w <a href="https://introcs.cs.princeton.edu/44st">podrozdziale 4.4</a> książki
 *  <i>Wprowadzenie do programowania w Javie</i> (Robert Sedgewick i Kevin Wayne). 
 *
 *  @author Robert Sedgewick
 *  @author Kevin Wayne
 *
 *  @param <Key> generyczny typ kluczy z tej tablicy symboli
 *  @param <Value> generyczny typ wartości z tej tablicy symboli
 */
public class ST<Key extends Comparable<Key>, Value> implements Iterable<Key> {

    private TreeMap<Key, Value> st;

    /**
     * Inicjuje pustą tablicę symboli.
     */
    public ST() {
        st = new TreeMap<Key, Value>();
    }


    /**
     * Zwraca wartość powiązaną z danym kluczem w tablicy symboli.
     *
     * @param  key klucz
     * @return wartość powiązaną z danym kluczem w tablicy symboli;
     *         {@code null}, jeśli klucz nie występuje w tablicy
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public Value get(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano get() dla klucza null");
        return st.get(key);
    }

    /**
     * Wstawia określoną parę klucz-wartość do tablicy symboli, zastępując
     * dawną wartość nową, jeśli dany klucz już znajduje się w tablicy.
     * Usuwa podany klucz (i powiązaną z nim wartość) z tablicy symboli,
     * jeżeli podana wartość to {@code null}.
     *
     * @param  key klucz
     * @param  val wartość
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public void put(Key key, Value val) {
        if (key == null) throw new IllegalArgumentException("wywołano put() dla klucza null");
        if (val == null) st.remove(key);
        else             st.put(key, val);
    }

    /**
     * Usuwa podany klucz i powiązaną z nim wartość z tablicy symboli
     * (jeśli klucz się w niej znajduje).
     *
     * @param  key klucz
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     * @deprecated zastąpione przez {@link #remove(Comparable key)}.
     */
    @Deprecated
    public void delete(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano delete() dla klucza null");
        st.remove(key);
    }

    /**
     * Usuwa podany klucz i powiązaną z nim wartość z tablicy symboli
     * (jeśli klucz się w niej znajduje).
     *
     * @param  key klucz
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public void remove(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano remove() dla klucza null");
        st.remove(key);
    }

    /**
     * Zwraca true, jeśli tablica symboli zawiera dany klucz
     *
     * @param  key klucz
     * @return {@code true}, jeśli tablica symboli zawiera klucz {@code key};
     *         {@code false} w przeciwnym razie
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public boolean contains(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano contains() dla klucza null");
        return st.containsKey(key);
    }

    /**
     * Zwraca liczbę par klucz-wartość w tablicy symboli
     *
     * @return liczbę par klucz-wartość w tablicy symboli
     */
    public int size() {
        return st.size();
    }

    /**
     * Zwraca true, jeśli tablica symboli jest pusta.
     *
     * @return {@code true}, jeśli tablica symboli jest pusta; {@code false} w przeciwnym razie
     */
    public boolean isEmpty() {
        return size() == 0;
    }

    /**
     * Zwraca wszystkie klucze z tablicy symboli
     * <p>
     * Do przechodzenia po wszystkich klucza tablicy symboli {@code st}
     * zastosuj konstrukcję foreach: {@code for (Key key : st.keys())}.
     *
     * @return wszystkie klucze z tablicy symboli
     */
    public Iterable<Key> keys() {
        return st.keySet();
    }

    /**
     * Zwraca wszystkie klucze z tablicy symboli.
     * Aby przejść po wszystkich kluczach z tablicy symboli {@code st} zastosuj
     * zapis foreach: {@code for (Key key : st)}.
     *
     * @return iterator dla wszystkich kluczy z tablicy symboli
     * @deprecated zastąpiona przez {@link #keys()}.
     */
    @Deprecated
    public Iterator<Key> iterator() {
        return st.keySet().iterator();
    }

    /**
     * Zwraca najmniejszy klucz z tablicy symboli
     *
     * @return najmniejszy klucz z tablicy symboli
     * @throws NoSuchElementException, jeśli tablica symboli jest pusta
     */
    public Key min() {
        if (isEmpty()) throw new NoSuchElementException("wywołano min() dla pustej tablicy symboli");
        return st.firstKey();
    }

    /**
     * Zwraca największy klucz z tablicy symboli
     *
     * @return największy klucz z tablicy symboli
     * @throws NoSuchElementException, jeśli tablica symboli jest pusta
     */
    public Key max() {
        if (isEmpty()) throw new NoSuchElementException("wywołano max() dla pustej tablicy symboli");
        return st.lastKey();
    }

    /**
     * Zwraca najmniejszy klucz z tablicy symboli większy lub równy względem {@code key}.
     *
     * @param  key klucz
     * @return najmniejszy klucz z tablicy symboli większy lub równy względem {@code key}
     * @throws NoSuchElementException, jeśli dany klucz nie istnieje
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public Key ceiling(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano ceiling() dla klucza null");
        Key k = st.ceilingKey(key);
        if (k == null) throw new NoSuchElementException("wszystkie klucze są mniejsze niż " + key);
        return k;
    }

    /**
     * Zwraca największy klucz z tablicy symboli mniejszy lub równy względem {@code key}.
     *
     * @param  key klucz
     * @return największy klucz z tablicy symboli mniejszy lub równy względem {@code key}
     * @throws NoSuchElementException, jeśli nie ma takiego klucza
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public Key floor(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano floor() dla klucza null");
        Key k = st.floorKey(key);
        if (k == null) throw new NoSuchElementException("wszystkie klucze są większe niż " + key);
        return k;
    }

    /**
     * Testy jednostkowe dla typu danych {@code ST}.
     */
    public static void main(String[] args) {
        ST<String, Integer> st = new ST<String, Integer>();
        for (int i = 0; !StdIn.isEmpty(); i++) {
            String key = StdIn.readString();
            st.put(key, i);
        }
        for (String s : st.keys())
            StdOut.println(s + " " + st.get(s));
    }
}