/******************************************************************************
 *  Kompilacja:  javac SET.java
 *  Wykonanie:    java SET
 *  Zależności: StdOut.java
 *  
 *  Implementacja zbioru z użyciem biblioteki TreeSet Javy.
 *  Duplikaty są niedozwolone.
 *
 *  % java SET
 *  128.112.136.11
 *  208.216.181.15
 *  null
 *
 ******************************************************************************/

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.TreeSet;

/**
 *  Klasa {@code SET} reprezentuje uporządkowany zbiór porównywalnych kluczy.
 *  Udostępnia standardowe metody <em>add</em>, <em>contains</em> i <em>delete</em>.
 *  Ponadto zawiera metody wyszukujące elementy w zbiorach uporządkowanych (<em>minimum</em>,
 *  <em>maximum</em>, <em>floor</em> i <em>ceiling</em>) oraz metody dla zbiorów
 *  (<em>union</em>, <em>intersection</em> i <em>equality</em>).
 *  <p>
 *  Choć ta implementacja obejmuje metodę {@code equals()}, nie obejmuje
 *  metody {@code hashCode()}, ponieważ zbiory są modyfikowalne.
 *  <p>
 *  W tej implementacji używane są drzewa BST. To wymaga, aby 
 *  typ używany dla kluczy implementował interfejs {@code Comparable}, metodę
 *  {@code compareTo()} i metodę porównującą dwa klucze. Nie są tu używane metody
 *  {@code equals()} ani {@code hashCode()}.
 *  Metody <em>add</em>, <em>contains</em>, <em>delete</em>, <em>minimum</em>,
 *  <em>maximum</em>, <em>ceiling</em> i <em>floor</em> w najgorszym razie
 *  działają w czasie logarytmicznym.
 *  Operacje <em>size</em>, i <em>isEmpty</em> działają w stałym czasie.
 *  Konstrukcja obiektów zajmuje stały czas.
 *  <p>  
 *  Dodatkową dokumentację znajdziesz w <a href="https://introcs.cs.princeton.edu/44st">podrozdziale 4.4</a> książki
 *  <i>Wprowadzenie do programowania w Javie</i> (Robert Sedgewick i Kevin Wayne). 
 *
 *  @author Robert Sedgewick
 *  @author Kevin Wayne
 *
 *  @param <Key> typ generyczny kluczy z tego zbioru
 */

public class SET<Key extends Comparable<Key>> implements Iterable<Key> {
    private TreeSet<Key> set;

    /**
     * Inicjuje pusty zbiór.
     */
    public SET() {
        set = new TreeSet<Key>();
    }

    /**
     * Inicjuje nowy zbiór będący niezależną kopią podanego zbioru.
     */
    public SET(SET<Key> x) {
        set = new TreeSet<Key>(x.set);
    }

    /**
     * Dodaje klucz do danego zbioru (jeśli klucz się w nim jeszcze nie znajduje).
     *
     * @param  key dodawany klucz
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public void add(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano add() dla klucza null");
        set.add(key);
    }


    /**
     * Zwraca true, jeśli zbiór zawiera dany klucz
     *
     * @param  key klucz
     * @return {@code true}, jeśli zbiór zawiera klucz {@code key};
     *         {@code false} w przeciwnym razie
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public boolean contains(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano contains() dla klucza null");
        return set.contains(key);
    }

    /**
     * Usuwa podany klucz z danego zbioru (jeśli zbiór zawiera ten klucz).
     *
     * @param  key klucz
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public void delete(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano delete() dla klucza null");
        set.remove(key);
    }

    /**
     * Zwraca liczbę kluczy w zbiorze.
     *
     * @return liczbę kluczy w zbiorze
     */
    public int size() {
        return set.size();
    }

    /**
     * Zwraca true, jeśli zbiór jest pusty
     *
     * @return {@code true} jeśli zbiór jest pusty;
     *         {@code false} w przeciwnym razie
     */
    public boolean isEmpty() {
        return size() == 0;
    }
 
    /**
     * Zwraca wszystkie klucze z danego zbioru jako iterator.
     * Aby przejść po wszystkich klucza zbioru {@code set}, zastosuj notację
     * foreach: {@code for (Key key : set)}.
     *
     * @return iterator przechodzący po wszystkich kluczach ze zbioru
     */
    public Iterator<Key> iterator() {
        return set.iterator();
    }

    /**
     * Zwraca największy klucz ze zbioru
     *
     * @return największy klucz ze zbioru
     * @throws NoSuchElementException, jeśli zbiór jest pusty
     */
    public Key max() {
        if (isEmpty()) throw new NoSuchElementException("wywołano max() dla pustego zbioru");
        return set.last();
    }

    /**
     * Zwraca najmniejszy klucz ze zbioru
     *
     * @return najmniejszy klucz ze zbioru
     * @throws NoSuchElementException, jeśli zbiór jest pusty
     */
    public Key min() {
        if (isEmpty()) throw new NoSuchElementException("wywołano min() dla pustego zbioru");
        return set.first();
    }


    /**
     * Zwraca najmniejszy klucz zbioru większy lub równy względem {@code key}.
     *
     * @param  key klucz
     * @return najmniejszy klucz zbioru większy lub równy względem {@code key}
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     * @throws NoSuchElementException, jeśli nie ma takiego klucza
     */
    public Key ceiling(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano ceiling() dla klucza null");
        Key k = set.ceiling(key);
        if (k == null) throw new NoSuchElementException("wszystkie klucze są mniejsze od " + key);
        return k;
    }

    /**
     * Zwraca największy klucz zbioru mniejszy lub równy względem {@code key}.
     *
     * @param  key klucz
     * @return największy klucz zbioru mniejszy lub równy względem {@code key}
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     * @throws NoSuchElementException, jeśli taki klucz nie istnieje
     */
    public Key floor(Key key) {
        if (key == null) throw new IllegalArgumentException("wywołano floor() dla klucza null");
        Key k = set.floor(key);
        if (k == null) throw new NoSuchElementException("wszystkie klucze są większe niż " + key);
        return k;
    }

    /**
     * Zwraca sumę danego zbioru i zbioru that.
     *
     * @param  that inny zbiór
     * @return sumę danego zbioru i zbioru that
     * @throws IllegalArgumentException, jeśli {@code that} to {@code null}
     */
    public SET<Key> union(SET<Key> that) {
        if (that == null) throw new IllegalArgumentException("wywołano union() z argumentem null");
        SET<Key> c = new SET<Key>();
        for (Key x : this) {
            c.add(x);
        }
        for (Key x : that) {
            c.add(x);
        }
        return c;
    }

    /**
     * Zwraca część wspólną danego zbioru i zbioru that.
     *
     * @param  that inny zbiór
     * @return część wspólną danego zbioru i zbioru that
     * @throws IllegalArgumentException, jeśli {@code that} to {@code null}
     */
    public SET<Key> intersects(SET<Key> that) {
        if (that == null) throw new IllegalArgumentException("wywołano intersects() dla argumentu null");
        SET<Key> c = new SET<Key>();
        if (this.size() < that.size()) {
            for (Key x : this) {
                if (that.contains(x)) c.add(x);
            }
        }
        else {
            for (Key x : that) {
                if (this.contains(x)) c.add(x);
            }
        }
        return c;
    }

    /**       
     * Porównuje dany zbiór ze zbiorem other.
     * <p>
     * Zauważ, że według tej metody dwa pusty zbiory są sobie równe,
     * nawet jeśli są sparametryzowane innymi typami.
     * Jest to zgodne z działaniem metody {@code equals()} 
     * z platformy Collections Javy.
     *       
     * @param  other inny zbiór
     * @return {@code true}, jeśli dany zbiór jest równy względem {@code other};
     *         {@code false} w przeciwnym razie
     */
    @SuppressWarnings("rawtypes")
    @Override
    public boolean equals(Object other) {
        if (other == this) return true;
        if (other == null) return false;
        if (other.getClass() != this.getClass()) return false;
        SET that = (SET) other;
        return this.set.equals(that.set);
    }

    /**
     * Ta operacja nie jest obsługiwana, ponieważ zbiory są modyfikowalne.
     *
     * @return nie zwraca wartości
     * @throws UnsupportedOperationException, jeśli zostanie wywołana
     */
    @Override
    public int hashCode() {
        throw new UnsupportedOperationException("hashCode() nie jest obsługiwana, ponieważ zbiory są modyfikowalne");
    }

    /**
     * Zwraca tekstową reprezentację zbioru.
     *
     * @return  tekstową reprezentację zbioru w nawiasach klamrowych i kolejnymi
     *         kluczami rozdzielanymi przecinkiem i spacją
     */
    @Override
    public String toString() {
        String s = set.toString();
        return "{ " + s.substring(1, s.length() - 1) + " }";
    }  

    /**
     * Testy jednostkowe typu danych {@code SET}.
     */
    public static void main(String[] args) {
        SET<String> set = new SET<String>();

        // Wstawianie kluczy
        set.add("www.cs.princeton.edu");
        set.add("www.cs.princeton.edu");    // Zastępowanie wcześniejszej wartości
        set.add("www.princeton.edu");
        set.add("www.math.princeton.edu");
        set.add("www.yale.edu");
        set.add("www.amazon.com");
        set.add("www.simpsons.com");
        set.add("www.stanford.edu");
        set.add("www.google.com");
        set.add("www.ibm.com");
        set.add("www.apple.com");
        set.add("www.slashdot.com");
        set.add("www.whitehouse.gov");
        set.add("www.espn.com");
        set.add("www.snopes.com");
        set.add("www.movies.com");
        set.add("www.cnn.com");
        set.add("www.iitb.ac.in");


        StdOut.println(set.contains("www.cs.princeton.edu"));
        StdOut.println(!set.contains("www.harvardsucks.com"));
        StdOut.println(set.contains("www.simpsons.com"));
        StdOut.println();

        StdOut.println("ceiling(www.simpsonr.com) = " + set.ceiling("www.simpsonr.com"));
        StdOut.println("ceiling(www.simpsons.com) = " + set.ceiling("www.simpsons.com"));
        StdOut.println("ceiling(www.simpsont.com) = " + set.ceiling("www.simpsont.com"));
        StdOut.println("floor(www.simpsonr.com)   = " + set.floor("www.simpsonr.com"));
        StdOut.println("floor(www.simpsons.com)   = " + set.floor("www.simpsons.com"));
        StdOut.println("floor(www.simpsont.com)   = " + set.floor("www.simpsont.com"));
        StdOut.println();


        // Wyświetla wszystkie klucze ze zbioru w porządku leksykograficznym
        for (String s : set) {
            StdOut.println(s);
        }

        StdOut.println();
        SET<String> set2 = new SET<String>(set);
        StdOut.println(set.equals(set2));
    }

}