/******************************************************************************
 *  Kompilacja:  javac Queue.java
 *  Wykonanie:    java Queue < input.txt
 *  Pliki z danymi:   https://introcs.cs.princeton.edu/43stack/tobe.txt  
 *
 *  Generyczna kolejka zaimplementowana za pomocą listy połączonej.
 *
 *  % java Queue < tobe.txt 
 *  to be or not to be (w kolejce pozostało: 2)
 *
 ******************************************************************************/

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 *  Klasa {@code Queue} reprezentuje kolejkę FIFO
 *  z generycznymi elementami.
 *  Obsługuje standardowe operacje <em>enqueue</em> i <em>dequeue</em>,
 *  a także metody do sprawdzania pierwszego elementu bez usuwania go z kolejki,
 *  sprawdzania, czy kolejka jest pusta, pobierania liczby elementów w kolejce, a
 *  także iterowania po elementach w porządku FIFO.
 *  <p>
 *  Ta implementacja używa listy połączonej z zagnieżdżoną klasą reprezentującą
 *  węzły tej listy.
 *  Operacje <em>enqueue</em>, <em>dequeue</em>, <em>peek</em>, <em>size</em> i <em>is-empty</em>
 *  działają w stałym czasie.
 *  <p>
 *  Dodatkową dokumentację znajdziesz w <a href="https://introcs.cs.princeton.edu/43stack">podrozdziale 4.3</a> książki
 *  <i>Wprowadzenie do programowania w Javie</i> (Robert Sedgewick i Kevin Wayne).
 *
 *  @author Robert Sedgewick
 *  @author Kevin Wayne
 *
 *  @param <Item> generyczny typ elementów kolejki
 */
public class Queue<Item> implements Iterable<Item> {
    private int n;         // Liczba elementów w kolejce
    private Node first;    // początek kolejki
    private Node last;     // koniec kolejki

    // klasa pomocnicza na potrzeby listy połączonej
    private class Node {
        private Item item;
        private Node next;
    }

    /**
     * Inicjuje pustę kolejkę.
     */
    public Queue() {
        first = null;
        last  = null;
        n = 0;
    }

    /**
     * Zwraca true, jeśli kolejka jest pusta.
     *
     * @return {@code true}, jeśli kolejka jest pusta; w przeciwnym razie zwraca {@code false}
     */
    public boolean isEmpty() {
        return first == null;
    }

    /**
     * Zwraca liczbę elementów w kolejce.
     *
     * @return liczbę elementów w kolejce
     */
    public int size() {
        return n;     
    }

    /**
     * Zwraca liczbę elementów w kolejce.
     *
     * @return liczbę elementów w kolejce
     */
    public int length() {
        return n;     
    }

    /**
     * Zwraca element najdawniej dodany do kolejki.
     *
     * @return element najdawniej dodany do kolejki
     * @throws NoSuchElementException, jeśli kolejka jest pusta
     */
    public Item peek() {
        if (isEmpty()) throw new NoSuchElementException("Kolejka pusta");
        return first.item;
    }

   /**
     * Dodaje element do kolejki.
     */
    public void enqueue(Item item) {
        Node oldlast = last;
        last = new Node();
        last.item = item;
        last.next = null;
        if (isEmpty()) first = last;
        else           oldlast.next = last;
        n++;
    }

    /**
     * Usuwa i zwraca najdawniej dodany do kolejki element.
     *
     * @return najdawniej dodany do kolejki element
     * @throws NoSuchElementException, jeśli kolejka jest pusta
     */
    public Item dequeue() {
        if (isEmpty()) throw new NoSuchElementException("kolejka pusta");
        Item item = first.item;
        first = first.next;
        n--;
        if (isEmpty()) last = null;   // unikanie zbędnych referencji
        return item;
    }

    /**
     * Zwraca ciąg znaków reprezentujący tę kolejkę.
     *
     * @return sekwencję rozdzielonych specjami elementów w kolejności FIFO 
     */
    public String toString() {
        StringBuilder s = new StringBuilder();
        for (Item item : this) {
            s.append(item);
            s.append(' ');
        }
        return s.toString();
    } 
 

    /**
     * Zwraca iterator iterujący po elementach kolejki w kolejności FIFO.
     *
     * @return iterator iterujący po elementach kolejki w kolejności FIFO
     */
    public Iterator<Item> iterator()  {
        return new ListIterator();  
    }

    // Iterator; nie implementuje opcjonalnej metody remove() 
    private class ListIterator implements Iterator<Item> {
        private Node current = first;

        public boolean hasNext()  { return current != null;                     }
        public void remove()      { throw new UnsupportedOperationException();  }

        public Item next() {
            if (!hasNext()) throw new NoSuchElementException();
            Item item = current.item;
            current = current.next; 
            return item;
        }
    }


    /**
     * Testy jednostkowe typu danych {@code Queue}.
     */
    public static void main(String[] args) {
        Queue<String> queue = new Queue<String>();
        while (!StdIn.isEmpty()) {
            String item = StdIn.readString();
            if (!item.equals("-")) queue.enqueue(item);
            else if (!queue.isEmpty()) StdOut.print(queue.dequeue() + " ");
        }
        StdOut.println("(w kolejce pozostało: " + queue.size() + ")");
    }
}