/******************************************************************************
 *  Kompilacja:  javac BST.java
 *  Wykonanie:    java BST
 *  Zależności: StdIn.java StdOut.java Queue.java
 *  Pliki z danymi:   https://introcs.cs.princeton.edu/44st/tinyST.txt  
 *
 *  Tablica symboli zaimplementowana za pomocą binarnego drzewa poszukiwań.
 * 
 *  % more tinyST.txt
 *  S E A R C H E X A M P L E
 *  
 *  % java BST < tinyST.txt
 *  A 8
 *  C 4
 *  E 12
 *  H 5
 *  L 11
 *  M 9
 *  P 10
 *  R 3
 *  S 0
 *  X 7
 *
 ******************************************************************************/

import java.util.NoSuchElementException;

/**
 *  Klasa {@code BST} reprezentuje uporządkowaną tablicę symboli z generycznymi
 *  parami klucz-wartość.
 *  Obsługuje standardowe metody <em>put</em>, <em>get</em>, <em>contains</em>,
 *  <em>delete</em>, <em>size</em> i <em>is-empty</em>.
 *  Udostępnia też metody do wyszukiwania danych w uporządkowanej kolekcji: <em>minimum</em>,
 *  <em>maximum</em>, <em>floor</em>, <em>select</em>, <em>ceiling</em>.
 *  Obejmuje też metodę <em>keys</em> do iterowania po wszystkich kluczach.
 *  Tablica symboli implementuje abstrakcję w postaci <em>tablicy asocjacyjnej</em>:
 *  gdy wartość jest wiązana z kluczem, który już znajduje się w tablicy symboli,
 *  zgodnie z konwencją dawna wartość jest zastępowana nową.
 *  Inaczej niż w klasie {@link java.util.Map} tu używana jest konwencja, że
 *  wartości nie mogą być równe {@code null}. Ustawienie wartości powiązanej z kluczem
 *  na {@code null} oznacza usunięcie klucza z tablicy symboli.
 *  <p>
 *  W tej implementacji używane jest (niezrównoważone) binarne drzewo poszukiwań. To wymaga, 
 *  by typ klucza implementował interfejs {@code Comparable} i wywoływał metodę
 *  {@code compareTo()} w celu porównywania dwóch elementów. Typ nie wywołuje metod
 *  {@code equals()} i {@code hashCode()}.
 *  Operacje <em>put</em>, <em>contains</em>, <em>remove</em>, <em>minimum</em>,
 *  <em>maximum</em>, <em>ceiling</em>, <em>floor</em>, <em>select</em> i
 *  <em>rank</em> działają w najgorszym razie w czasie liniowym
 *  (jeśli drzewo jest niezrównoważone).
 *  Operacje <em>size</em> i <em>is-empty</em> działają w stałym czasie.
 *  Tworzenie obiektu zachodzi w całym czasie.
 *  <p>
 *  Dodatkową dokumentację znajdziesz w <a href="http://algs4.cs.princeton.edu/44st">podrozdział 4.4</a> książki
 *  <i>Wprowadzenie do programowania w Javie</i> autorstwa Roberta Sedgewicka i Kevina Wayne'a.
 *  Inne implementacje: {@link ST}, {@link BinarySearchST},
 *  {@link SequentialSearchST} i {@link HashST}.
 *
 *  @author Robert Sedgewick
 *  @author Kevin Wayne
 */
public class BST<Key extends Comparable<Key>, Value> {
    private Node root;             // korzeń drzewa BST

    private class Node {
        private final Key key;       // posortowane według klucza
        private Value val;           // powiązane dane
        private Node left, right;    // lewe i prawe poddrzewo
        private int size;            // liczba węzłów w poddrzewie

        public Node(Key key, Value val, int size) {
            this.key = key;
            this.val = val;
            this.size = size;
        }
    }

    /**
     * Inicjuje pustą tablicę symboli.
     */
    public BST() {
    }

    /**
     * Zwraca true, jeśli tablica symboli jest pusta.
     * @return {@code true}, jeśli tablica symboli jest pusta; w przeciwnym razie zwraca {@code false}
     */
    public boolean isEmpty() {
        return size() == 0;
    }

    /**
     * Zwraca liczbę par klucz-wartość w tej tablicy symboli.
     * @return liczbę par klucz-wartość w tej tablicy symboli
     */
    public int size() {
        return size(root);
    }

    // Zwraca liczbę par klucz-wartość w drzewie BST z korzeniem w x
    private int size(Node x) {
        if (x == null) return 0;
        else return x.size;
    }

    /**
     * Czy dana tablica symboli zawiera dany klucz?
     *
     * @param  key klucz
     * @return {@code true}, jeśli tablica symboli zawiera {@code key}; w przeciwnym razie zwraca
     *         {@code false}
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public boolean contains(Key key) {
        if (key == null) throw new IllegalArgumentException("argument metody contains() to null");
        return get(key) != null;
    }

    /**
	 * Zwraca wartość powiązaną z danym kluczem.
     *
     * @param  key klucz
     * @return wartość powiązaną z danym kluczem, jeśli klucz znajduje sięw tablicy symboli;
     *         w przeciwnym razie zwraca {@code null}
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public Value get(Key key) {
        return get(root, key);
    }

    private Value get(Node x, Key key) {
        if (x == null) return null;
        int cmp = key.compareTo(x.key);
        if      (cmp < 0) return get(x.left, key);
        else if (cmp > 0) return get(x.right, key);
        else              return x.val;
    }

    /**
     * Wstawia określoną parę klucz-wartość do tablicy symboli, zastępując dawną wartość nową,
     * jeśli tablica symboli już zawiera dany klucz.
     * Usuwa określony klucz (i powiązaną wartość), jeśli
     * podana wartość to {@code null}.
     *
     * @param  key klucz
     * @param  val wartość
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public void put(Key key, Value val) {
        if (key == null) throw new IllegalArgumentException("pierwszy argument metody put() to null");
        if (val == null) {
            delete(key);
            return;
        }
        root = put(root, key, val);
        assert check();
    }

    private Node put(Node x, Key key, Value val) {
        if (x == null) return new Node(key, val, 1);
        int cmp = key.compareTo(x.key);
        if      (cmp < 0) x.left  = put(x.left,  key, val);
        else if (cmp > 0) x.right = put(x.right, key, val);
        else              x.val   = val;
        x.size = 1 + size(x.left) + size(x.right);
        return x;
    }


    /**
     * Usuwa najmniejszy klucz i powiązaną wartość z tablicy symboli.
     *
     * @throws NoSuchElementException, jeśli tablica symboli jest pusta
     */
    public void deleteMin() {
        if (isEmpty()) throw new NoSuchElementException("Tablica jest pusta");
        root = deleteMin(root);
        assert check();
    }

    private Node deleteMin(Node x) {
        if (x.left == null) return x.right;
        x.left = deleteMin(x.left);
        x.size = size(x.left) + size(x.right) + 1;
        return x;
    }

    /**
     * Usuwa największy klucz i powiązaną wartość z tablicy symboli.
     *
     * @throws NoSuchElementException, jeśli tablica symboli jest pusta
     */
    public void deleteMax() {
        if (isEmpty()) throw new NoSuchElementException("Tablica jest pusta");
        root = deleteMax(root);
        assert check();
    }

    private Node deleteMax(Node x) {
        if (x.right == null) return x.left;
        x.right = deleteMax(x.right);
        x.size = size(x.left) + size(x.right) + 1;
        return x;
    }

    /**
     * Usuwa określony klucz i powiązaną wartość z tablicy symboli
     * (jeśli klucz występuje w tablicy symboli).    
     *
     * @param  key klucz
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public void delete(Key key) {
        if (key == null) throw new IllegalArgumentException("argument metody delete() to null");
        root = delete(root, key);
        assert check();
    }

    private Node delete(Node x, Key key) {
        if (x == null) return null;

        int cmp = key.compareTo(x.key);
        if      (cmp < 0) x.left  = delete(x.left,  key);
        else if (cmp > 0) x.right = delete(x.right, key);
        else { 
            if (x.right == null) return x.left;
            if (x.left  == null) return x.right;
            Node t = x;
            x = min(t.right);
            x.right = deleteMin(t.right);
            x.left = t.left;
        } 
        x.size = size(x.left) + size(x.right) + 1;
        return x;
    } 


    /**
     * Zwraca najmniejszy klucz z tablicy symboli.
     *
     * @return najmniejszy klucz w tablicy symboli
     * @throws NoSuchElementException, jeśli tablica symboli jest pusta
     */
    public Key min() {
        if (isEmpty()) throw new NoSuchElementException("wywołano min() dla pustej tablicy symboli");
        return min(root).key;
    } 

    private Node min(Node x) { 
        if (x.left == null) return x; 
        else                return min(x.left); 
    } 

    /**
     * Zwraca największy kluczy w tablicy symboli
     *
     * @return największy kluczy w tablicy symboli
     * @throws NoSuchElementException, jeśli tablica jest pusta
     */
    public Key max() {
        if (isEmpty()) throw new NoSuchElementException("wywołano max() dla pustej tablicy symboli");
        return max(root).key;
    } 

    private Node max(Node x) {
        if (x.right == null) return x; 
        else                 return max(x.right); 
    } 

    /**
     * Zwraca największy klucz w tablicy symboli mniejszy lub równy {@code key}.
     *
     * @param  key klucz
     * @return największy klucz w tablicy symboli mniejszy lub równy {@code key}
     * @throws NoSuchElementException, jeśli taki klucz nie istnieje
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public Key floor(Key key) {
        if (key == null) throw new IllegalArgumentException("argument metody floor() to null");
        if (isEmpty()) throw new NoSuchElementException("wywołano floor() dla pustej tablicy");
        Node x = floor(root, key);
        if (x == null) return null;
        else return x.key;
    } 

    private Node floor(Node x, Key key) {
        if (x == null) return null;
        int cmp = key.compareTo(x.key);
        if (cmp == 0) return x;
        if (cmp <  0) return floor(x.left, key);
        Node t = floor(x.right, key); 
        if (t != null) return t;
        else return x; 
    } 

    /**
     * Zwraca najmniejszy klucz z tablicy symboli większy lub równy {@code key}.
     *
     * @param  key klucz
     * @return najmniejszy klucz z tablicy symboli większy lub równy {@code key}
     * @throws NoSuchElementException, jeśli taki klucz nie istnieje
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public Key ceiling(Key key) {
        if (key == null) throw new IllegalArgumentException("argument metody ceiling() to null");
        if (isEmpty()) throw new NoSuchElementException("wywołano ceiling() dla pustej tablicy");
        Node x = ceiling(root, key);
        if (x == null) return null;
        else return x.key;
    }

    private Node ceiling(Node x, Key key) {
        if (x == null) return null;
        int cmp = key.compareTo(x.key);
        if (cmp == 0) return x;
        if (cmp < 0) { 
            Node t = ceiling(x.left, key); 
            if (t != null) return t;
            else return x; 
        } 
        return ceiling(x.right, key); 
    } 

    /**
     * Zwraca k-ty najmniejszy klucz z tablicy symboli.
     *
     * @param  k pozycja
     * @return k-ty najmniejszy klucz z tablicy symboli
     * @throws IllegalArgumentException, jeśli {@code k} nie występuje między 0 a
     *        <em>N</em> &minus; 1
     */
    public Key select(int k) {
        if (k < 0 || k >= size()) throw new IllegalArgumentException();
        Node x = select(root, k);
        return x.key;
    }

    // Zwraca klucz z pozycji k. 
    private Node select(Node x, int k) {
        if (x == null) return null; 
        int t = size(x.left); 
        if      (t > k) return select(x.left,  k); 
        else if (t < k) return select(x.right, k-t-1); 
        else            return x; 
    } 

    /**
     * Zwraca liczbę kluczy w tablicy symboli mniejszych niż {@code key}.
     *
     * @param  key klucz
     * @return liczba kluczy w tablicy symboli mniejszych niż {@code key}
     * @throws IllegalArgumentException, jeśli {@code key} to {@code null}
     */
    public int rank(Key key) {
        if (key == null) throw new IllegalArgumentException("argument metody rank() to null");
        return rank(key, root);
    } 

	// Liczba kluczy w poddrzewie mniejszych od danego
    private int rank(Key key, Node x) {
        if (x == null) return 0; 
        int cmp = key.compareTo(x.key); 
        if      (cmp < 0) return rank(key, x.left); 
        else if (cmp > 0) return 1 + size(x.left) + rank(key, x.right); 
        else              return size(x.left); 
    } 

    /**
     * Zwraca wszystkie klucze z tablicy symboli jako obiekt {@code Iterable}.
     * Do iterowania po wszystkich kluczach tablicy symboli {@code st}
     * zastosuj notację foreach: {@code for (Key key : st.keys())}.
     *
     * @return wszystkie klucze z tablicy symboli
     */
    public Iterable<Key> keys() {
        return rangeSearch(min(), max());
    }

    /**
     * Zwraca wszystkie klucze z tablicy symboli z określonego przedziału
     * jako obiekt typu {@code Iterable}.
     *
     * @return wszystkie klucze z tablicy symboli spomiędzy {@code lo} 
     *         (włącznie) a {@code hi} (z pominięciem)
     * @throws IllegalArgumentException, jeśli  {@code lo} lub {@code hi}
     *         to {@code null}
     */
    public Iterable<Key> rangeSearch(Key lo, Key hi) {
        if (lo == null) throw new IllegalArgumentException("pierwszy argument metody keys() to null");
        if (hi == null) throw new IllegalArgumentException("drugi argument metody keys() to null");

        Queue<Key> queue = new Queue<Key>();
        keys(root, queue, lo, hi);
        return queue;
    } 

    private void keys(Node x, Queue<Key> queue, Key lo, Key hi) { 
        if (x == null) return; 
        int cmplo = lo.compareTo(x.key); 
        int cmphi = hi.compareTo(x.key); 
        if (cmplo < 0) keys(x.left, queue, lo, hi); 
        if (cmplo <= 0 && cmphi >= 0) queue.enqueue(x.key); 
        if (cmphi > 0) keys(x.right, queue, lo, hi); 
    } 

    /**
     * Zwraca liczbę kluczy z podanego przedziału.
     *
     * @return liczbę kluczy pomiędzy wartościami {@code lo} 
     *         (włącznie) i {@code hi} (z pominięciem)
     * @throws IllegalArgumentException, jeśli {@code lo} lub {@code hi}
     *         to {@code null}
     */
    public int rangeCount(Key lo, Key hi) {
        if (lo == null) throw new IllegalArgumentException("pierwszy argument metody size() to null");
        if (hi == null) throw new IllegalArgumentException("drugi argument metody size() to null");

        if (lo.compareTo(hi) > 0) return 0;
        if (contains(hi)) return rank(hi) - rank(lo) + 1;
        else              return rank(hi) - rank(lo);
    }

    /**
     * Zwraca wysokość drzewa BST (na potrzeby debugowania).
     *
     * @return wysokość drzewa BST (wysokość drzewa z jednym węzłem to 0)
     */
    public int height() {
        return height(root);
    }
    private int height(Node x) {
        if (x == null) return -1;
        return 1 + Math.max(height(x.left), height(x.right));
    }

    /**
     * Zwraca klucze z drzewa BST poziomami (na potrzeby debugowania).
     *
     * @return klucze drzewa BST poziomami
     */
    public Iterable<Key> levelOrder() {
        Queue<Key> keys = new Queue<Key>();
        Queue<Node> queue = new Queue<Node>();
        queue.enqueue(root);
        while (!queue.isEmpty()) {
            Node x = queue.dequeue();
            if (x == null) continue;
            keys.enqueue(x.key);
            queue.enqueue(x.left);
            queue.enqueue(x.right);
        }
        return keys;
    }

  /*************************************************************************
    *  Sprawdzanie integralności struktury danych BST.
    ***************************************************************************/
    private boolean check() {
        if (!isBST())            StdOut.println("Drzewo nie jest symetryczne");
        if (!isSizeConsistent()) StdOut.println("Liczba elementów w poddrzewach nie jest spójna");
        if (!isRankConsistent()) StdOut.println("Pozycje nie są spójne");
        return isBST() && isSizeConsistent() && isRankConsistent();
    }

    // Czy kolejność elementów w drzewie jest symetryczna?
    // Uwaga: ten test gwarantuje też, że ta struktura danych jest drzewem binarnym, ponieważ elementy są odpowiednie uporządkowane
    private boolean isBST() {
        return isBST(root, null, null);
    }

    // czy drzewo z korzeniem x to drzewo BST z wszystkimi kluczami między min a max?
    // Autor: eleganckie rozwiązanie Boba Dondero
    private boolean isBST(Node x, Key min, Key max) {
        if (x == null) return true;
        if (min != null && x.key.compareTo(min) <= 0) return false;
        if (max != null && x.key.compareTo(max) >= 0) return false;
        return isBST(x.left, min, x.key) && isBST(x.right, x.key, max);
    } 

    // Czy pola określające wielkość mają odpowiednie wartości?
    private boolean isSizeConsistent() { return isSizeConsistent(root); }
    private boolean isSizeConsistent(Node x) {
        if (x == null) return true;
        if (x.size != size(x.left) + size(x.right) + 1) return false;
        return isSizeConsistent(x.left) && isSizeConsistent(x.right);
    } 

    // sprawdzanie, czy pozycje są spójne
    private boolean isRankConsistent() {
        for (int i = 0; i < size(); i++)
            if (i != rank(select(i))) return false;
        for (Key key : keys())
            if (key.compareTo(select(rank(key))) != 0) return false;
        return true;
    }


    /**
     * Testy jednostkowe dla typu danych {@code BST}.
     */
    public static void main(String[] args) { 
        BST<String, Integer> st = new BST<String, Integer>();
        for (int i = 0; !StdIn.isEmpty(); i++) {
            String key = StdIn.readString();
            st.put(key, i);
        }

        for (String s : st.levelOrder())
            StdOut.println(s + " " + st.get(s));

        StdOut.println();

        for (String s : st.keys())
            StdOut.println(s + " " + st.get(s));
    }
}