/******************************************************************************
 *  Kompilacja:  javac Gaussian.java
 *  Wykonywanie:    java Gaussian x mu sigma
 *
 *  Funkcja obliczająca funkcję gęstości prawdopodobieństwa i
 *  dystrybuantę dla rozkładu Gaussa 
 *
 *  % java Gaussian 820 1019 209
 *  0.17050966869132111
 *
 *  % java Gaussian 1500 1019 209
 *  0.9893164837383883
 *
 *  % java Gaussian 1500 1025 231
 *  0.9801220907365489
 *
 *  Błąd przybliżenia jest mniejszy niż 8 * 10^(-16).
 *  Źródło: Evaluating the Normal Distribution autorstwa George'a Marsaglia.
 *  http://www.jstatsoft.org/v11/a04/paper
 *
 ******************************************************************************/

public class Gaussian {

    // zwraca pdf(x) = standardowa funkcja gęstości prawdopodobieństwa dla rozkładu Gaussa
    public static double pdf(double x) {
        return Math.exp(-x*x / 2) / Math.sqrt(2 * Math.PI);
    }

    // zwraca pdf(x, mu, signma) = funkcja gęstości prawdopodobieństwa dla rozkładu Gaussa ze średnią mu i odchyleniem standardowym sigma
    public static double pdf(double x, double mu, double sigma) {
        return pdf((x - mu) / sigma) / sigma;
    }

    // zwraca cdf(z) = standardowa dystrybuanta dla rozkładu Gaussa na podstawie przybliżenia Taylora
    public static double cdf(double z) {
        if (z < -8.0) return 0.0;
        if (z >  8.0) return 1.0;
        double sum = 0.0, term = z;
        for (int i = 3; sum + term != sum; i += 2) {
            sum  = sum + term;
            term = term * z * z / i;
        }
        return 0.5 + sum * pdf(z);
    }

    // zwraca cdf(z, mu, sigma) = dystrybuanta dla rozkładu Gaussa ze średnią mu i odchyleniem stand. sigma
    public static double cdf(double z, double mu, double sigma) {
        return cdf((z - mu) / sigma);
    } 

    // Oblicza "z" takie że cdf(z) = y (z wykorzystaniem wyszukiwania binarnego)
    public static double inverseCDF(double y) {
        return inverseCDF(y, 0.00000001, -8, 8);
    } 

    // wyszukiwanie binarne
    private static double inverseCDF(double y, double delta, double lo, double hi) {
        double mid = lo + (hi - lo) / 2;
        if (hi - lo < delta) return mid;
        if (cdf(mid) > y) return inverseCDF(y, delta, lo, mid);
        else              return inverseCDF(y, delta, mid, hi);
    }


    // zwraca phi(x) = standardowa funkcja gęstości prawdopodobieństwa dla rozkładu Gaussa
    @Deprecated
    public static double phi(double x) {
        return pdf(x);
    }

    // zwraca phi(x, mu, signma) = funkcja gęstości prawdopodobieństwa dla rozkładu Gaussa ze średnią mu i odch. stand. sigma
    @Deprecated
    public static double phi(double x, double mu, double sigma) {
        return pdf(x, mu, sigma);
    }

    // zwraca Phi(z) = standardowa dystrybuanta dla rozkładu Gaussa na podstawie przybliżenia Taylora
    @Deprecated
    public static double Phi(double z) {
        return cdf(z);
    }

    // return Phi(z, mu, sigma) = dystrybuanta dla rozkładu Gaussa ze średnią mu i odchyleniem stand. sigma
    @Deprecated
    public static double Phi(double z, double mu, double sigma) {
        return cdf(z, mu, sigma);
    } 

    // Oblicza "z" takie że Phi(z) = y (z użyciem wyszukiwania binarnego)
    @Deprecated
    public static double PhiInverse(double y) {
        return inverseCDF(y);
    } 

    // klient testowy
    public static void main(String[] args) {
        double z     = Double.parseDouble(args[0]);
        double mu    = Double.parseDouble(args[1]);
        double sigma = Double.parseDouble(args[2]);
        StdOut.println(cdf(z, mu, sigma));
        double y = cdf(z);
        StdOut.println(inverseCDF(y));
    }
}