/**
  Listingi z rozdziału 3.
  Autor: Luis Atencio
*/

"use strict";

QUnit.module('Rozdział 3.');

const _ = require('lodash');
const R = require('ramda');

const Person = require('../model/Person.js').Person;
const Address = require('../model/Address.js').Address;

var p1 = new Person('111-11-1111', 'Haskell', 'Curry', 1900, new Address('USA'));	
var p2 = new Person('222-22-2222', 'Barkley', 'Rosser', 1907, new Address('Grecja'));
var p3 = new Person('333-33-3333', 'John', 'von Neumann', 1903, new Address('Węgry'));
var p4 = new Person('444-44-4444', 'Alonzo', 'Church', 1903, new Address('USA'));

var persons = [p1, p2, p3, p4];

QUnit.test("Metoda reduce", function () {
	let result = _(persons).reduce((stat, person) => {
		const country = person.address.country;
		stat[country] = _.isUndefined(stat[country]) ? 1 :
		stat[country] + 1;
		return stat;
	}, {});	    

	assert.deepEqual(result, {
		'USA' : 2,
		'Grecja' : 1,
		'Węgry': 1
	});	
});

QUnit.test("Łączenie map i reduce", function () {
	const getCountry = person => person.address.country;
	const gatherStats = (stat, criteria) => {
		stat[criteria] = _.isUndefined(stat[criteria]) ? 1 :
			stat[criteria] + 1;
		return stat;
	};
	let result = _(persons).map(getCountry).reduce(gatherStats, {});
	assert.deepEqual(result, { USA: 2, Grecja: 1, Węgry: 1 });		
});


QUnit.test("Łączenie map i reduce z soczewkami", function () {
	const cityPath = ['address','city'];
	const cityLens = R.lens(R.path(cityPath), R.assocPath(cityPath));
	const gatherStats = (stat, criteria) => {
		stat[criteria] = _.isUndefined(stat[criteria]) ? 1 :
			stat[criteria] + 1;
		return stat;
	};
	
	let result = _(persons).map(R.view(cityLens)).reduce(gatherStats, {});	
	assert.deepEqual(result, { null: 4 });	// DO ZROBIENIA	
});  

QUnit.test("Poprawne lub niepoprawne", function () {
	const isNotValid = val => _.isUndefined(val) || _.isNull(val);
	const notAllValid = args => _(args).some(isNotValid);
	assert.ok(notAllValid (['string', 0, null, undefined])); //-> false
	assert.ok(!notAllValid (['string', 0, {}]));             //-> true	

	const isValid = val => !_.isUndefined(val) && !_.isNull(val);
	const allValid = args => _(args).every(isValid);
	assert.ok(!allValid(['string', 0, null])); //-> false
	assert.ok(allValid(['string', 0, {}]));    //-> true
});  

QUnit.test("Metoda filter", function () {
	const isValid = val => !_.isUndefined(val) && !_.isNull(val);
	const fullname = person => person.fullname;
	let result = _([p1, p2, p3, null]).filter(isValid).map(fullname).value();
	assert.equal(result.length, 3);
});  

QUnit.test("Osoby urodzone w 1903", function () {
	const bornIn1903 = person => person.birthYear === 1903;
	const fullname = person => person.fullname;

	let result = _(persons).filter(bornIn1903).map(fullname).join(' i ');
	assert.equal(result, 'John von Neumann i Alonzo Church');	
});  

QUnit.test("Przetwarzanie tablic za pomocą biblioteki Lodash", function () {
	let names = ['alonzo church', 'Haskell curry', 'stephen_kleene',
				 'John Von Neumann', 'stephen_kleene'];
	
	const isValid = val => !_.isUndefined(val) && !_.isNull(val);

	var result = _.chain(names).filter(isValid).map(s => s.replace(/_/, ' '))
		.uniq().map(_.startCase).sort().value();
	
	assert.deepEqual(result, ['Alonzo Church', 'Haskell Curry', 'John Von Neumann', 'Stephen Kleene']);			 
});  


QUnit.test("Rejestrowanie statystyk", function () {
	const gatherStats = function (stat, country) {
		if(!isValid(stat[country])) {
			stat[country] = {'name': country, 'count': 0};
		}
		stat[country].count++;
		return stat;
	};
	const isValid = val => !_.isUndefined(val) && !_.isNull(val);	
	const getCountry = person => person.address.country;	
	let result = _(persons).map(getCountry).reduce(gatherStats, {});
	assert.deepEqual(result, 
		{ US:      { name: 'USA', count: 2 },
  		  Greece:  { name: 'Grecja', count: 1 },
  		  Hungary: { name: 'Węgry', count: 1 } 
  		}
  	);		
});  

// Połowa rozdziału - do tablicy dodawane są nowe obiekty.
// Tu dodatkowe dane znajdują się w tablicy persons2, choć w rozdziale nadal jest ona nazywana persons
var persons2 = _(persons).map(R.identity);
var p5 = new Person('555-55-5555', 'David', 'Hilbert', 1903, new Address('Niemcy'));
persons2.push(p5);

var p6 = new Person('666-66-6666', 'Alan', 'Turing', 1912, new Address('Anglia'));
persons2.push(p6);

var p7 = new Person('777-77-7777', 'Stephen', 'Kleene', 1909, new Address('USA'));
persons2.push(p7);

QUnit.test("Leniwe łańcuchy funkcji", function () {

	const gatherStats = function (stat, country) {
		if(!isValid(stat[country])) {
			stat[country] = {'name': country, 'count': 0};
		}
		stat[country].count++;
		return stat;
	};
	const isValid = val => !_.isUndefined(val) && !_.isNull(val);	

	let result = _.chain(persons)
		.filter(isValid)
		.map(_.property('address.country'))
		.reduce(gatherStats, {})
		.values()
		.sortBy('count')
		.reverse()
		.first()
		.value()
		.name; //-> 'USA'

	assert.equal(result, 'USA');			
});  


QUnit.test("JavaScript w stylu SQL-a", function () {

	_.mixin({'select': _.map, 
			 'from': _.chain,
			 'where': _.filter
			});			 

	let result = _.from(persons)
		.where(p => p.birthYear > 1900 && p.address.country !== 'USA')
		.sortBy(['_firstname'])
		.select(rec => rec.firstname)
		.value();		

	assert.deepEqual(result, ['Barkley', 'John']);	
});  

QUnit.test("Rekurencyjne dodawanie", function () {
	function sum(arr) {
		if(_.isEmpty(arr)) {
			return 0;
		}
		return _.first(arr) + sum(_.tail(arr));
	};
	
	assert.equal(sum([]), 0); //-> 0
	assert.equal(sum([1,2,3,4,5,6,7,8,9]), 45); //->45
});  

QUnit.test("Poruszanie się po drzewie", function () {
	const Node = require('./model/Node.js').Node;	
	const Tree = require('./model/Tree.js').Tree;	
	
	// Tworzenie wszystkich węzłów i dodawanie nowych danych
	const church = new Node(p4);
	const rosser = new Node(p2);
	const turing = new Node(p6);
	const kleene = new Node(p7);
	const nelson = new Node(new Person('123-23-2345', 'Nels', 'Nelson'))
	const constable = new Node(new Person('123-23-6778', 'Robert', 'Constable'));
	const mendelson = new Node(new Person('123-23-3454', 'Elliot', 'Mendelson')); 
	const sacks = new Node(new Person('454-76-3434', 'Gerald', 'Sacks'));
	const gandy = new Node(new Person('454-78-3432','Robert', 'Gandy'));

	// Tworzenie struktury drzewiastej
	church.append(rosser).append(turing).append(kleene);
	kleene.append(nelson).append(constable);
	rosser.append(mendelson).append(sacks);
	turing.append(gandy);

	// Używanie struktury Tree w celu zastosowania operacji map do wszystkich węzłów
	let newTree = Tree.map(church, p => p.fullname);
	assert.deepEqual(newTree.toArray(), ['Alonzo Church', 'Barkley Rosser', 'Elliot Mendelson', 
		'Gerald Sacks', 'Alan Turing', 'Robert Gandy', 'Stephen Kleene', 'Nels Nelson', 'Robert Constable']);	
});

