#!/usr/bin/env node

'use strict';

const util = require('util');
const express = require('express');
const http = require('http');
const path = require('path');
const fs   = require('fs');
const favicon = require('serve-favicon');
const logger = require('morgan');
const FileStreamRotator = require('file-stream-rotator');
const cookieParser = require('cookie-parser');
const bodyParser = require('body-parser');
const session = require('express-session')
const FileStore = require('session-file-store')(session);
const passportSocketIo = require("passport.socketio");

const routes = require('./routes/index');
const users  = require('./routes/users'); 
const notes  = require('./routes/notes');

const log   = require('debug')('notes:server');
const error = require('debug')('notes:error');

process.on('uncaughtException', function(err) {
  error("Nastąpiła awaria!!! - "+ (err.stack || err));
});

const sessionCookie = 'notes.sid';
const sessionSecret = 'keyboard mouse';
const sessionStore  = new FileStore({ path: process.env.NOTES_SESSIONS_DIR ? process.env.NOTES_SESSIONS_DIR : "sessions" });

var app = express();

/**
 * Tworzenie serwera HTTP.
 */

var server = http.createServer(app);
var io = require('socket.io')(server);

io.use(passportSocketIo.authorize({
  cookieParser: cookieParser, 
  key:          sessionCookie,
  secret:       sessionSecret,
  store:        sessionStore
}));

/**
 * Pobieranie portu ze środowiska i zapisywanie go w platformie Express.
 */

var port = normalizePort(process.env.PORT || '3000');
app.set('port', port);

// Konfigurowanie silnika widoków
app.set('views', path.join(__dirname, 'views'));
app.set('view engine', 'ejs');

var accessLogStream;
if (process.env.REQUEST_LOG_FILE) {
    var logDirectory = path.dirname(process.env.REQUEST_LOG_FILE);
    fs.existsSync(logDirectory) || fs.mkdirSync(logDirectory);
    accessLogStream = FileStreamRotator.getStream({
      filename: process.env.REQUEST_LOG_FILE,
      frequency: 'daily',
      verbose: false
    });
}

// Przenieś kod poza komentarz po umieszczeniu pliku favicon.ico w katalogu /public
//app.use(favicon(path.join(__dirname, 'public', 'favicon.ico')));
app.use(logger(process.env.REQUEST_LOG_FORMAT || 'dev', {
    stream: accessLogStream ? accessLogStream : process.stdout
}));
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: false }));
app.use(cookieParser());
app.use(express.static(path.join(__dirname, 'public')));
// app.use('/vendor/bootstrap/css', express.static(path.join(__dirname, 'cyborg')));
app.use('/vendor/bootstrap/css', express.static(path.join(__dirname, 'bower_components', 'bootstrap', 'dist', 'css')));
app.use('/vendor/bootstrap/fonts', express.static(path.join(__dirname, 'bower_components', 'bootstrap', 'dist', 'fonts')));
app.use('/vendor/bootstrap/js', express.static(path.join(__dirname, 'bower_components', 'bootstrap', 'dist', 'js')));
app.use('/vendor/jquery', express.static(path.join(__dirname, 'bower_components', 'jquery', 'dist')));

app.use(session({
  store: sessionStore,
  secret: sessionSecret,
  key: sessionCookie,
  resave: true,
  saveUninitialized: true
}));

users.initPassport(app);

app.use('/', routes);
app.use('/users', users.router);
app.use('/notes', notes);

routes.socketio(io);
notes.socketio(io);

// Przechwytywanie błędu 404 i przekazywanie go do funkcji obsługi błędów
app.use(function(req, res, next) {
  var err = new Error('Nie znaleziono');
  err.status = 404;
  next(err);
});

// Funkcje obsługi błędów

// Funkcja obsługi błędów dla środowiska programistycznego.
// Wyświetla stos błędów
if (app.get('env') === 'development') {
  app.use(function(err, req, res, next) {
    // util.log(err.message);
    res.status(err.status || 500);
    error((err.status || 500) +' '+ error.message);
    res.render('error', {
      message: err.message,
      error: err
    });
  });
}

// Funkcja obsługi błędów dla środowiska produkcyjnego.
// Nie ujawnia użytkownikom stosu błędów
app.use(function(err, req, res, next) {
  // util.log(err.message);
  res.status(err.status || 500);
  error((err.status || 500) +' '+ error.message);
  res.render('error', {
    message: err.message,
    error: {}
  });
});


module.exports = app;


/**
 * Oczekiwanie na żądania w podanym porcie dla wszystkich interfejsów sieciowych.
 */

server.listen(port);
server.on('error', onError);
server.on('listening', onListening);


/**
 * Przekształcanie portu na liczbę, łańcuch znaków lub wartość false.
 */

function normalizePort(val) {
  var port = parseInt(val, 10);

  if (isNaN(port)) {
    // Potok nazwany
    return val;
  }

  if (port >= 0) {
    // Numer portu
    return port;
  }

  return false;
}

/**
 * Odbiornik zdarzenia error serwera HTTP.
 */

function onError(error) {
  if (error.syscall !== 'listen') {
    throw error;
  }

  var bind = typeof port === 'string'
    ? 'Pipe ' + port
    : 'Port ' + port;

  // Obsługa konkretnych błędów z użyciem zrozumiałych komunikatów
  switch (error.code) {
    case 'EACCES':
      console.error(bind + ' wymaga podwyższonych uprawnień');
      process.exit(1);
      break;
    case 'EADDRINUSE':
      console.error(bind + ' jest już w użyciu');
      process.exit(1);
      break;
    default:
      throw error;
  }
}

/**
 * Odbiornik zdarzenia listening serwera HTTP.
 */

function onListening() {
  var addr = server.address();
  var bind = typeof addr === 'string'
    ? 'pipe ' + addr
    : 'port ' + addr.port;
  log('Oczekiwanie na zdarzenia pod adresem ' + bind);
}
