// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using System.Web.UI.WebControls;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.MultiHandleSliderExtender Model
    /// </summary>
    /// <TestComponent Name="Slider">
    ///   <ToolkitType>AjaxControlToolkit.MultiHandleSliderExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.MultiHandleSlider.MultiHandleSlider_Common</CommonTestSuite>
    /// </TestComponent>
    public class MultiHandleSliderBehavior : Behavior<HtmlInputElement>
    {
        /// <summary>
        /// Minimum
        /// </summary>
        public BehaviorProperty<double> Minimum
        {
            get { return _minimum; }
        }
        private BehaviorProperty<double> _minimum;

        /// <summary>
        /// Maximum
        /// </summary>
        public BehaviorProperty<double> Maximum
        {
            get { return _maximum; }
        }
        private BehaviorProperty<double> _maximum;

        /// <summary>
        /// RailCssClass
        /// </summary>
        public BehaviorProperty<string> CssClass
        {
            get { return _cssClass; }
        }
        private BehaviorProperty<string> _cssClass;

        /// <summary>
        /// EnableHandleAnimation
        /// </summary>
        public BehaviorProperty<bool> EnableHandleAnimation
        {
            get { return _enableHandleAnimation; }
        }
        private BehaviorProperty<bool> _enableHandleAnimation;

        /// <summary>
        /// Steps
        /// </summary>
        public BehaviorProperty<int> Steps
        {
            get { return _steps; }
        }
        private BehaviorProperty<int> _steps;

        /// <summary>
        /// Orientation
        /// </summary>
        public BehaviorProperty<SliderOrientation> Orientation
        {
            get { return _orientation; }
        }
        private BehaviorProperty<SliderOrientation> _orientation;

        /// <summary>
        /// Decimals
        /// </summary>
        public BehaviorProperty<int> Decimals
        {
            get { return _decimals; }
        }
        private BehaviorProperty<int> _decimals;

        /// <summary>
        /// BoundControlID
        /// </summary>
        public ReferenceBehaviorProperty<HtmlInputElement> BoundControl
        {
            get { return _boundControl; }
        }
        private ReferenceBehaviorProperty<HtmlInputElement> _boundControl;

        /// <summary>
        /// Length
        /// </summary>
        public BehaviorProperty<Unit> Length
        {
            get { return _length; }
        }
        private BehaviorProperty<Unit> _length;

        /// <summary>
        /// RaiseChangeOnlyOnMouseUp
        /// </summary>
        public BehaviorProperty<bool> RaiseChangeOnlyOnMouseUp
        {
            get { return _raiseChangeOnlyOnMouseUp; }
        }
        private BehaviorProperty<bool> _raiseChangeOnlyOnMouseUp;

        /// <summary>
        /// TooltipText
        /// </summary>
        public BehaviorProperty<string> TooltipText
        {
            get { return _tooltipText; }
        }
        private BehaviorProperty<string> _tooltipText;

        public BehaviorProperty<double> SliderValue
        {
            get { return _value; }
        }
        private BehaviorProperty<double> _value;

        public BehaviorProperty<string> SliderValues
        {
            get { return _values; }
        }
        private BehaviorProperty<string> _values;

        /// <summary>
        /// AjaxControlToolkit.MultiHandleSliderExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public MultiHandleSliderBehavior(HtmlInputElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _minimum = BehaviorProperty<double>.CreateProperty(this, "minimum");
            _maximum = BehaviorProperty<double>.CreateProperty(this, "maximum");
            _cssClass = BehaviorProperty<string>.CreateProperty(this, "cssClass");
            _enableHandleAnimation = BehaviorProperty<bool>.CreateProperty(this, "enableHandleAnimation");
            _steps = BehaviorProperty<int>.CreateProperty(this, "steps");
            _orientation = BehaviorProperty<SliderOrientation>.CreateProperty(this, "orientation");
            _decimals = BehaviorProperty<int>.CreateProperty(this, "decimals");
            _boundControl = ReferenceBehaviorProperty<HtmlInputElement>.CreateProperty(this, "boundControlID");
            _length = BehaviorProperty<Unit>.CreateProperty(this, "length", 
                delegate(object obj) 
                { 
                    string length = (obj == null)? "0px" : obj.ToString();
                    return Unit.Parse(length);
                });
            _raiseChangeOnlyOnMouseUp = BehaviorProperty<bool>.CreateProperty(this, "raiseChangeOnlyOnMouseUp");
            _tooltipText = BehaviorProperty<string>.CreateProperty(this, "tooltipText");

            // Value as in single-handle scenarios only
            _value = BehaviorProperty<double>.CreateProperty(this, "Value");

            // Value for all handles
            _values = BehaviorProperty<string>.CreateProperty(this, "values");
        }

        /// <summary>
        /// validate that the slider state is correct
        /// </summary>
        /// <param name="expectedSliderValue"></param>
        public void AssertSliderValue(double expectedSliderValue)
        {
            SliderValue.Invalidate();
            double currentVal = SliderValue.Value;
            ToolkitAssert.All(
                    delegate { Assert.AreEqual(expectedSliderValue, currentVal); },
                    delegate { Assert.IsTrue(Convert.ToDouble(currentVal) <= this.Maximum.Value); },
                    delegate { Assert.IsTrue(Convert.ToDouble(currentVal) >= this.Minimum.Value); },
                    delegate 
                    { 
                        Assert.AreEqual(currentVal, double.Parse(this.Element.GetAttributes().Value)); 
                    },
                    delegate
                        {
                        if (BoundControl.Reference != null)
                        {
                            Assert.AreEqual(expectedSliderValue, BoundControlValue);
                        }
                    });
        }

        public double BoundControlValue
        {
            get
            {
                return double.Parse(BoundControl.Reference.GetAttributes().Value.Trim());
            }
            set
            {
                BoundControl.Reference.SetText(value.ToString());
            }
        }
    }
}