// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.Globalization;
using System.Web.UI.WebControls;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.CalendarExtender Model
    /// </summary>
    /// <TestComponent Name="Calendar">
    ///   <ToolkitType>AjaxControlToolkit.CalendarExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.Calendar.Calendar_Common</CommonTestSuite>
    /// </TestComponent>
    public class CalendarBehavior : Behavior<HtmlInputElement>
    {
        /// <summary>
        /// CssClass
        /// </summary>
        public BehaviorProperty<string> CssClass
        {
            get { return _cssClass; }
        }
        private BehaviorProperty<string> _cssClass;

        /// <summary>
        /// Format
        /// </summary>
        public BehaviorProperty<string> Format
        {
            get { return _format; }
        }
        private BehaviorProperty<string> _format;

        /// <summary>
        /// Enabled
        /// </summary>
        public BehaviorProperty<bool> Enabled
        {
            get { return _enabled; }
        }
        private BehaviorProperty<bool> _enabled;

        /// <summary>
        /// Animated
        /// </summary>
        public BehaviorProperty<bool> Animated
        {
            get { return _animated; }
        }
        private BehaviorProperty<bool> _animated;

        /// <summary>
        /// FirstDayOfWeek
        /// </summary>
        public BehaviorProperty<FirstDayOfWeek> FirstDayOfWeek
        {
            get { return _firstDayOfWeek; }
        }
        private BehaviorProperty<FirstDayOfWeek> _firstDayOfWeek;

        /// <summary>
        /// PopupPosition
        /// </summary>
        public BehaviorProperty<CalendarPosition> PopupPosition
        {
            get { return _popupPosition; }
        }
        private BehaviorProperty<CalendarPosition> _popupPosition;

        /// <summary>
        /// SelectedDate
        /// </summary>
        public DateBehaviorProperty SelectedDate
        {
            get { return _selectedDate; }
        }
        private DateBehaviorProperty _selectedDate;

        /// <summary>
        /// VisibleDate
        /// </summary>
        public DateBehaviorProperty VisibleDate
        {
            get { return _visibleDate; }
        }
        private DateBehaviorProperty _visibleDate;

        /// <summary>
        /// TodaysDate
        /// </summary>
        public DateBehaviorProperty TodaysDate
        {
            get { return _todaysDate; }
        }
        private DateBehaviorProperty _todaysDate;

        /// <summary>
        /// PopupElement
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> PopupElement
        {
            get { return _popupElement; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _popupElement;

        /// <summary>
        /// PreviousArrow
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> PreviousArrow
        {
            get { return _previousArrow; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _previousArrow;

        /// <summary>
        /// NextArrow
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> NextArrow
        {
            get { return _nextArrow; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _nextArrow;

        /// <summary>
        /// Title
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Title
        {
            get { return _title; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _title;

        /// <summary>
        /// Today
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Today
        {
            get { return _today; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _today;

        /// <summary>
        /// Button
        /// </summary>
        public ReferenceBehaviorProperty<HtmlInputElement> Button
        {
            get { return _button; }
        }
        private ReferenceBehaviorProperty<HtmlInputElement> _button;

        /// <summary>
        /// Container
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Container
        {
            get { return _container; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _container;

        /// <summary>
        /// Header
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Header
        {
            get { return _header; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _header;

        /// <summary>
        /// Body
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Body
        {
            get { return _body; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _body;

        /// <summary>
        /// Days
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Days
        {
            get { return _days; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _days;

        /// <summary>
        /// DaysTable
        /// </summary>
        public ReferenceBehaviorProperty<HtmlTableElement> DaysTable
        {
            get { return _daysTable; }
        }
        private ReferenceBehaviorProperty<HtmlTableElement> _daysTable;

        /// <summary>
        /// DaysTableHeader
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> DaysTableHeader
        {
            get { return _daysTableHeader; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _daysTableHeader;

        /// <summary>
        /// DaysTableBody
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> DaysTableBody
        {
            get { return _daysTableBody; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _daysTableBody;

        /// <summary>
        /// Months
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Months
        {
            get { return _months; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _months;

        /// <summary>
        /// MonthsTable
        /// </summary>
        public ReferenceBehaviorProperty<HtmlTableElement> MonthsTable
        {
            get { return _monthsTable; }
        }
        private ReferenceBehaviorProperty<HtmlTableElement> _monthsTable;

                /// <summary>
        /// MonthsBody
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> MonthsBody
        {
            get { return _monthsBody; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _monthsBody;

        /// <summary>
        /// Years
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Years
        {
            get { return _years; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _years;

        /// <summary>
        /// YearsBody
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> YearsBody
        {
            get { return _yearsBody; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _yearsBody;


        /// <summary>
        /// YearsTable
        /// </summary>
        public ReferenceBehaviorProperty<HtmlTableElement> YearsTable
        {
            get { return _yearsTable; }
        }
        private ReferenceBehaviorProperty<HtmlTableElement> _yearsTable;

        /// <summary>
        /// Mode
        /// </summary>
        public BehaviorProperty<string> Mode
        {
            get { return _mode; }
        }
        private BehaviorProperty<string> _mode;

        /// <summary>
        /// IsSelectedDateChanging
        /// </summary>
        public BehaviorProperty<bool> IsSelectedDateChanging
        {
            get { return _isSelectedDateChanging; }
        }
        private BehaviorProperty<bool> _isSelectedDateChanging;

        /// <summary>
        /// IsOpen
        /// </summary>
        public BehaviorProperty<bool> IsOpen
        {
            get { return _isOpen; }
        }
        private BehaviorProperty<bool> _isOpen;

        /// <summary>
        /// IsAnimating
        /// </summary>
        public BehaviorProperty<bool> IsAnimating
        {
            get { return _isAnimating; }
        }
        private BehaviorProperty<bool> _isAnimating;

        /// <summary>
        /// Width
        /// </summary>
        public BehaviorProperty<int> Width
        {
            get { return _width; }
        }
        private BehaviorProperty<int> _width;

        /// <summary>
        /// Height
        /// </summary>
        public BehaviorProperty<int> Height
        {
            get { return _height; }
        }
        private BehaviorProperty<int> _height;

        /// <summary>
        /// HourOffsetForDaylightSavingsTime
        /// </summary>
        public BehaviorProperty<int> HourOffsetForDaylightSavingsTime
        {
            get { return _hourOffsetForDaylightSavingsTime; }
        }
        private BehaviorProperty<int> _hourOffsetForDaylightSavingsTime;

        /// <summary>
        /// Showing
        /// </summary>
        public BehaviorEvent Showing
        {
            get { return _showing; }
        }
        private BehaviorEvent _showing;

        /// <summary>
        /// Shown
        /// </summary>
        public BehaviorEvent Shown
        {
            get { return _shown; }
        }
        private BehaviorEvent _shown;

        /// <summary>
        /// Hiding
        /// </summary>
        public BehaviorEvent Hiding
        {
            get { return _hiding; }
        }
        private BehaviorEvent _hiding;

        /// <summary>
        /// Hidden
        /// </summary>
        public BehaviorEvent Hidden
        {
            get { return _hidden; }
        }
        private BehaviorEvent _hidden;

        /// <summary>
        /// DateSelectionChanged
        /// </summary>
        public BehaviorEvent DateSelectionChanged
        {
            get { return _dateSelectionChanged; }
        }
        private BehaviorEvent _dateSelectionChanged;

        /// <summary>
        /// Culture used to create and display the calendar
        /// </summary>
        public CultureInfo Culture
        {
            get { return _culture; }
            set { _culture = value; }
        }
        private CultureInfo _culture = CultureInfo.CurrentCulture;

        /// <summary>
        /// PopupBehavior
        /// </summary>
        public PopupBehavior PopupBehavior
        {
            get
            {
                if (_popupBehavior == null)
                {
                    _popupBehavior = new PopupBehavior(_popupElement.Reference, BehaviorID + "_popupBehavior", BehaviorReferenceExpression + "._popupBehavior", Page);
                }
                return _popupBehavior;
            }
        }
        private PopupBehavior _popupBehavior;

        /// <summary>
        /// AjaxControlToolkit.CalendarExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public CalendarBehavior(HtmlInputElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _cssClass = BehaviorProperty<string>.CreateProperty(this, "cssClass");
            _format = BehaviorProperty<string>.CreateProperty(this, "format");
            _enabled = BehaviorProperty<bool>.CreateProperty(this, "enabled");
            _animated = BehaviorProperty<bool>.CreateProperty(this, "animated");
            _firstDayOfWeek = BehaviorProperty<FirstDayOfWeek>.CreateProperty(this, "firstDayOfWeek");
            _popupPosition = BehaviorProperty<CalendarPosition>.CreateProperty(this, "popupPosition");
            _mode = BehaviorProperty<string>.CreateField(this, "_mode");
            _isSelectedDateChanging = BehaviorProperty<bool>.CreateField(this, "_selectedDateChanging");
            _isOpen = BehaviorProperty<bool>.CreateField(this, "_isOpen");
            _isAnimating = BehaviorProperty<bool>.CreateField(this, "_isAnimating");
            _width = BehaviorProperty<int>.CreateField(this, "_width");
            _height = BehaviorProperty<int>.CreateField(this, "_height");
            _hourOffsetForDaylightSavingsTime = BehaviorProperty<int>.CreateField(this, "_hourOffsetForDst");

            string getDate = "(function() {{{{ " +
                "var d = {{0}}.get_{0}(); " +
                "return !d ? null : " +
                    "(d.getUTCMonth() + 1) + '/' + d.getUTCDate() + '/' + d.getUTCFullYear(); ' GMT'" +
            "}}}})()";
            _selectedDate = new DateBehaviorProperty(this, "selectedDate", ClientMemberType.Custom, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate,
                null, string.Format(getDate, "selectedDate"), "{0}.set_selectedDate(new Date({1}))", null);
            _visibleDate = new DateBehaviorProperty(this, "visibleDate", ClientMemberType.Custom, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate,
                null, string.Format(getDate, "visibleDate"), "{0}.set_visibleDate(new Date({1}))", null);
            _todaysDate = new DateBehaviorProperty(this, "todaysDate", ClientMemberType.Custom, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate,
                null, string.Format(getDate, "todaysDate"), "{0}.set_todaysDate(new Date({1}))", null);
            
            _popupElement = CreateElementReference<HtmlElement>("_popupDiv");
            _previousArrow = CreateElementReference<HtmlElement>("_prevArrow");
            _nextArrow = CreateElementReference<HtmlElement>("_nextArrow");
            _title = CreateElementReference<HtmlElement>("_title");
            _today = CreateElementReference<HtmlElement>("_today");
            _button = CreateElementReference<HtmlInputElement>("_button");
            _container = CreateElementReference<HtmlElement>("_container");
            _header = CreateElementReference<HtmlElement>("_header");
            _body = CreateElementReference<HtmlElement>("_body");
            _days = CreateElementReference<HtmlElement>("_days");
            _daysTable = CreateElementReference<HtmlTableElement>("_daysTable");
            _daysTableHeader = CreateElementReference<HtmlElement>("_daysTableHeader");
            _daysTableBody = CreateElementReference<HtmlElement>("_daysBody");
            _months = CreateElementReference<HtmlElement>("_months");
            _monthsTable = CreateElementReference<HtmlTableElement>("_monthsTable");
            _monthsBody = CreateElementReference<HtmlElement>("_monthsBody");
            _years = CreateElementReference<HtmlElement>("_years");
            _yearsTable = CreateElementReference<HtmlTableElement>("_yearsTable");
            _yearsBody = CreateElementReference<HtmlElement>("_yearsBody");

            _showing = BehaviorEvent<object>.CreateEvent(this, "showing");
            _shown = BehaviorEvent<object>.CreateEvent(this, "shown");
            _hiding = BehaviorEvent<object>.CreateEvent(this, "hiding");
            _hidden = BehaviorEvent<object>.CreateEvent(this, "hidden");
            _dateSelectionChanged = BehaviorEvent<object>.CreateEvent(this, "dateSelectionChanged");
        }

        /// <summary>
        /// Create a reference to an element in the Calendar
        /// </summary>
        /// <typeparam name="T">Type of element</typeparam>
        /// <param name="variable">Variable containing element</param>
        /// <returns>ReferenceBehaviorProperty</returns>
        private ReferenceBehaviorProperty<T> CreateElementReference<T>(string variable)
            where T : HtmlElement
        {
            string reference = "{0}." + variable;
            return ReferenceBehaviorProperty<T>.CreateField(this, "id", ReadStrategy.Demand, WriteStrategy.Immediate, "{0}", reference + " ? " + reference + ".id : null", reference + ".id = {1}", null);
        }

        /// <summary>
        /// Show the calendar
        /// </summary>
        public void Show()
        {
            ShowOrHide(true);
        }

        /// <summary>
        /// Hide the calendar
        /// </summary>
        public void Hide()
        {
            ShowOrHide(false);
        }

        /// <summary>
        /// Show or hide the calendar
        /// </summary>
        /// <param name="show">Whether it should be shown</param>
        private void ShowOrHide(bool show)
        {
            // Do nothing if we're already correct
            if (_isOpen.Value == show)
            {
                return;
            }

            // Open/close by either focusing/blurring the textbox or clicking
            // the button depending on the mode we're in
            if (string.IsNullOrEmpty(_button.Value))
            {
                if (show)
                {
                    Element.Focus();
                }
                else
                {
                    PopupBehavior.Hidden.Register();
                    Element.Blur();
                    PopupBehavior.Hidden.Wait(10);
                }
            }
            else
            {
                if (!show)
                {
                    PopupBehavior.Hidden.Register();
                }

                _button.Reference.Click();

                if (!show)
                {
                    PopupBehavior.Hidden.Wait(10);
                }
            }

            // Verify that we're still open
            _isOpen.Invalidate();
            if (show)
            {
                Common.Wait(Page, 10, delegate { _isOpen.Invalidate(); Assert.IsTrue(_isOpen.Value); return true; });
            }
            Assert.AreEqual(show, _isOpen.Value);
        }

        /// <summary>
        /// Get elements for the days in the calendar
        /// </summary>
        /// <returns>Elements for the days in the calendar</returns>
        public HtmlElement[,] GetDayElements()
        {
            // Get a reference to the table's body
            HtmlElement element = _daysTableBody.Reference;
            if (element == null)
            {
                throw new InvalidOperationException("DaysTableBody has not been created yet!");
            }

            // Get the rows 
            List<HtmlTableRowElement> rows = new List<HtmlTableRowElement>();
            foreach (HtmlElement dayRow in element.ChildElements)
            {
                HtmlTableRowElement row = dayRow as HtmlTableRowElement;
                if (row != null)
                {
                    rows.Add(row);
                }
            }

            // Copy the cells into an array
            HtmlElement[,] cells = new HtmlElement[rows.Count, 7];
            for (int i = 0; i < rows.Count; i++)
            {
                for (int j = 0; j < 7; j++)
                {
                    cells[i, j] = rows[i].Cells[j].ChildElements[0];
                }
            }

            return cells;
        }

        /// <summary>
        /// Get the text for the days in the calendar
        /// </summary>
        /// <returns>Text for the elements in the calendar</returns>
        public string[,] GetDayText()
        {
            object result = Page.ExecuteScript("(function() { " +
                "var b = " + BehaviorReferenceExpression + "; " +
                "if (!b._daysBody) { return null; } " +
                "var rows = b._daysBody.childNodes; " +
                "var result = { }; " +
                "result.rows = rows.length; " +
                "result.columns = rows[0].childNodes.length; " +
                "for (var i = 0; i < rows.length; i++) { " +
                    "var cells = rows[i].childNodes; " +
                    "for (var j = 0; j < cells.length; j++) { " +
                        "result[i + ',' + j] = cells[j].childNodes[0].innerHTML; " +
                    "} " +
                "} " +
                "return result; " +
            "})()");
            if (result == null)
            {
                throw new InvalidOperationException("DaysTableBody has not been created yet!");
            }

            int rows = Common.GetJson<int>(result, "rows");
            int columns = Common.GetJson<int>(result, "columns");
            string[,] text = new string[rows, columns];
            for (int i = 0; i < rows; i++)
            {
                for (int j = 0; j < columns; j++)
                {
                    text[i, j] = Common.GetJson<string>(result, string.Format("{0},{1}", i, j));
                }
            }
            return text;
        }

        /// <summary>
        /// Determine the offset for the first day of the month
        /// </summary>
        /// <param name="date">Date in the month</param>
        /// <returns>Number of days before the first day of the month displayed in the calendar</returns>
        public int GetFirstDayOfMonthOffset(DateTime date)
        {
            // CultureInfo.CurrentCulture.DateTimeFormat.FirstDayOfWeek
            DateTime firstDayOfMonth = new DateTime(date.Year, date.Month, 1);
            int calendarFirstDayOfWeek = ((_firstDayOfWeek.Value == System.Web.UI.WebControls.FirstDayOfWeek.Default) ?
                (int) _culture.DateTimeFormat.FirstDayOfWeek :
                (int) _firstDayOfWeek.Value);
            int offset = ((int) firstDayOfMonth.DayOfWeek - calendarFirstDayOfWeek) % 7;
            if (offset <= 0)
            {
                offset += 7;
            }
            return offset;
        }

        /// <summary>
        /// Verify that the calendar's dates are correct for the expected month and year
        /// </summary>
        /// <param name="date">Date in the month to verify the calendar dates for</param>
        public void AssertCalendarDates(DateTime date)
        {
            // Set the date
            _selectedDate.Value = date.ToLongDateString();
            Show();

            // Determine where the calendar should start displaying dates
            DateTime firstDayOfMonth = new DateTime(date.Year, date.Month, 1);
            DateTime current = firstDayOfMonth.AddDays(-GetFirstDayOfMonthOffset(date));

            // Get the text of each date
            string[,] text = GetDayText();
            Assert.IsNotNull(text);
            Assert.AreEqual(2, text.Rank);
            Assert.AreNotEqual(0, text.GetLength(0));
            Assert.AreNotEqual(0, text.GetLength(1));
            
            // Make sure the text for each date is correct
            for (int i = 0; i < text.GetLength(0); i++)
            {
                for (int j = 0; j < text.GetLength(1); j++)
                {
                    int dateValue;
                    Assert.IsTrue(int.TryParse(text[i,j], out dateValue));
                    Assert.AreEqual(current.Day, dateValue);

                    // Move to the next date
                    current = current.AddDays(1);
                }
            }
        }
    }
}