// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Text;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    public static class ToolkitAssert
    {
        /// <summary>
        /// The Assertion delegate is used by ToolkitAssert.All to test multiple
        /// assertions at once via lazy evaluation.
        /// </summary>
        public delegate void Assertion();

        /// <summary>
        /// Test multiple assertions at once and combine any failures into a
        /// single assertion to provide a detailed explanation.
        /// </summary>
        /// <param name="assertions">Assertions to test</param>
        /// <example>
        /// The following example verifies several failing assertions at once:
        /// <code>ToolkitAssert.All(
        ///     delegate() { Assert.IsTrue(1 > 2); },
        ///     delegate() { Assert.AreEqual(1 + 1, 3); });
        ///     delegate() { Assert.StringIsNullOrEmpty("Fail!"); });</code>
        /// </example>
        public static void All(params Assertion[] assertions)
        {
            if (assertions == null)
            {
                return;
            }

            // Test all of the assertions and track any that fail
            List<WebTestingException> failures = new List<WebTestingException>();
            foreach (Assertion assert in assertions)
            {
                try { assert(); }
                catch (WebTestingException ex)
                {
                    failures.Add(ex);
                }
            }

            // Merge any failures into a single exception
            if (failures.Count > 0)
            {
                StringBuilder message = new StringBuilder();
                message.AppendFormat("{0} failed assertions!", failures.Count);
                for (int i = 0; i < failures.Count; i++)
                {
                    message.AppendFormat("  {0}. {1}", i + 1, failures[i].Message);
                }
                throw new WebTestingException(message.ToString());
            }
        }

        /// <summary>
        /// Assert the offsetWidth/offsetHeight of the specified HtmlElement
        /// </summary>
        /// <param name="element">element</param>
        /// <param name="expectedOffsetSize">expected size</param>
        public static void OffsetSize(HtmlElement element, Size expectedOffsetSize)
        {
            Size actualOffsetSize = Common.GetBounds(element).Size;
            Assert.AreEqual(expectedOffsetSize.Width, actualOffsetSize.Width);
            Assert.AreEqual(expectedOffsetSize.Height, actualOffsetSize.Height);
        }
    }
}
