// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.DropWatcherExtender Model
    /// </summary>
    /// <TestComponent Name="ReorderList">
    ///   <ToolkitType>AjaxControlToolkit.DropWatcherExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.ReorderList.ReorderList_Common</CommonTestSuite>
    /// </TestComponent>
    public class DropWatcherBehavior : Behavior<HtmlElement>
    {
        ///// <summary>
        ///// AcceptedDataTypes
        ///// </summary>
        //public BehaviorProperty<string> AcceptedDataTypes
        //{
        //    get { return _acceptedDataTypes; }
        //}
        //private BehaviorProperty<string> _acceptedDataTypes;

        /// <summary>
        /// ArgReplaceString
        /// </summary>
        public BehaviorProperty<string> ArgReplaceString
        {
            get { return _argReplaceString; }
        }
        private BehaviorProperty<string> _argReplaceString;

        /// <summary>
        /// ArgSuccessString
        /// </summary>
        public BehaviorProperty<string> ArgSuccessString
        {
            get { return _argSuccessString; }
        }
        private BehaviorProperty<string> _argSuccessString;

        /// <summary>
        /// ArgErrorString
        /// </summary>
        public BehaviorProperty<string> ArgErrorString
        {
            get { return _argErrorString; }
        }
        private BehaviorProperty<string> _argErrorString;

        /// <summary>
        /// ArgContextString
        /// </summary>
        public BehaviorProperty<string> ArgContextString
        {
            get { return _argContextString; }
        }
        private BehaviorProperty<string> _argContextString;

        /// <summary>
        /// CallbackCssStyle
        /// </summary>
        public BehaviorProperty<string> CallbackCssStyle
        {
            get { return _callbackCssStyle; }
        }
        private BehaviorProperty<string> _callbackCssStyle;

        /// <summary>
        /// DataType
        /// </summary>
        public BehaviorProperty<string> DataType
        {
            get { return _dataType; }
        }
        private BehaviorProperty<string> _dataType;

        /// <summary>
        /// DragMode
        /// </summary>
        public BehaviorProperty<int> DragMode
        {
            get { return _dragMode; }
        }
        private BehaviorProperty<int> _dragMode;

        ///// <summary>
        ///// DropLayoutElement
        ///// </summary>
        //public BehaviorProperty<string> DropLayoutElement
        //{
        //    get { return _dropLayoutElement; }
        //}
        //private BehaviorProperty<string> _dropLayoutElement;

        /// <summary>
        /// PostBackCode
        /// </summary>
        public BehaviorProperty<string> PostBackCode
        {
            get { return _postBackCode; }
        }
        private BehaviorProperty<string> _postBackCode;

        /// <summary>
        /// Items
        /// </summary>
        public BehaviorProperty<IList<DraggableListItemBehavior>> Items
        {
            get { return _items; }
        }
        private BehaviorProperty<IList<DraggableListItemBehavior>> _items;

        /// <summary>
        /// ReorderComplete event
        /// </summary>
        public BehaviorEvent ReorderComplete
        {
            get { return _reorderComplete; }
        }
        private BehaviorEvent _reorderComplete;

        /// <summary>
        /// AjaxControlToolkit.DropWatcherExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public DropWatcherBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            //_acceptedDataTypes = BehaviorProperty<string>.CreateProperty(this, "acceptedDataTypes");
            _argReplaceString = BehaviorProperty<string>.CreateProperty(this, "argReplaceString");
            _argSuccessString = BehaviorProperty<string>.CreateProperty(this, "argSuccessString");
            _argErrorString = BehaviorProperty<string>.CreateProperty(this, "argErrorString");
            _argContextString = BehaviorProperty<string>.CreateProperty(this, "argContextString");
            _callbackCssStyle = BehaviorProperty<string>.CreateProperty(this, "callbackCssStyle");
            _dataType = BehaviorProperty<string>.CreateProperty(this, "dragDataType");
            _dragMode = BehaviorProperty<int>.CreateProperty(this, "dragMode");
            //_dropLayoutElement = BehaviorProperty<string>.CreateProperty(this, "dropCueTemplate");
            _postBackCode = BehaviorProperty<string>.CreateProperty(this, "postbackCode");
            _items = BehaviorProperty<IList<DraggableListItemBehavior>>.CreateCustomProperty(this, null, "(function() {{ " +
                    "var b = {0}; " +
                    "if (!b._childList) {{ b._saveChildOrder(); }} " +
                    "return b._childList.join(' '); " +
                "}})()",
                "throw 'Items property is read-only!'", ConvertItems);
            _reorderComplete = BehaviorEvent<object>.CreateEvent(this, "reorderComplete");
        }

        /// <summary>
        /// Convert the value into a list of DraggableListItemBehaviors
        /// </summary>
        private IList<DraggableListItemBehavior> ConvertItems(object value)
        {
            List<DraggableListItemBehavior> items = new List<DraggableListItemBehavior>();
            string identifiers = value as string;
            if (!string.IsNullOrEmpty(identifiers))
            {
                foreach (string id in identifiers.Split(' '))
                {
                    HtmlElement element = Page.Elements.Find(id);
                    Assert.IsNotNull(element, "Failed to DraggableListItemBehavior \"{0}\" in DropWatcherBehavior \"{1}\"!", id, BehaviorID);
                    items.Add(new DraggableListItemBehavior(element, id, Page));
                }
            }
            return items;
        }

        /// <summary>
        /// Reorder the items in the list
        /// </summary>
        /// <param name="startIndex">Starting index of item to drag</param>
        /// <param name="endIndex">Index to drag the item to</param>
        public void Reorder(int startIndex, int endIndex)
        {
            Reorder(startIndex, endIndex, false);
        }

        /// <summary>
        /// Reorder the items in the list
        /// </summary>
        /// <param name="startIndex">Starting index of item to drag</param>
        /// <param name="endIndex">Index to drag the item to</param>
        /// <param name="skipPostack">Skip the postbacks</param>
        public void Reorder(int startIndex, int endIndex, bool skipPostack)
        {
            object result = Page.ExecuteScript("(function() { " +
                "var b = " + BehaviorReferenceExpression + "; " +
                "return { " +
                    "start: $get(b.getItem(" + startIndex + ")).innerHTML, " +
                    "end: $get(b.getItem(" + endIndex + ")).innerHTML " +
                "} " +
            "})()");
            string startContent = Common.GetJson<string>(result, "start");
            string originalContent = Common.GetJson<string>(result, "end");

            Page.ExecuteScript(BehaviorReferenceExpression + ".doReorder(" + startIndex + ", " + endIndex + ", " + skipPostack.ToString().ToLower() + "), true");
 
            // TODO: Currently we can't check for postbacks here...
            // _reorderComplete.Wait(10);

            string newContent = Page.ExecuteScript("$get(" + BehaviorReferenceExpression + ".getItem(" + endIndex + ")).innerHTML") as string;
            Assert.AreEqual(startContent, newContent);
        }

        /// <summary>
        /// Get the number of items in a ReorderList
        /// </summary>
        /// <param name="behavior">ReorderList</param>
        /// <returns>Number of items</returns>
        public int GetItemsLength()
        {
            int length = _items.Value.Count;
            if (length > 0 && _items.Value[length - 1].Element.Id.EndsWith("Insert"))
            {
                length--;
            }
            return length;
        }
    }
}