// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit.Testing.Client;
using AjaxControlToolkit.Testing.Client.MaskedEdit;

/// <summary>
/// Validate MaskedEdit works fine when the mask type is set to number.
/// </summary>
/// <TestSuite>
///   <Path>MaskedEdit\Number.aspx</Path>
///   <ComponentType>AjaxControlToolkit.Testing.Client.MaskedEditBehavior</ComponentType>
///   <WrapperType>AjaxControlToolkit.Testing.Client.MaskedEdit.NumberPage</WrapperType>
/// </TestSuite>
[WebTestClass]
[WebTestTag("MaskedEdit.Number")]
public class MaskedEdit_Number : AjaxControlToolkit.Testing.Client.MaskedEdit.MaskedEdit_Common
{
    public override IEnumerable<MaskedEditScenario> MaskedEdits
    {
        get 
        {
            NumberPage page = new NumberPage();
            yield return new MaskedEditScenario(page.HundredKMaskedEditExtender, "___.___,__", page.HundredKMaskedEditValidator);
            yield return new MaskedEditScenario(page.NumberTextBoxMaskedEditExtender, "___-___-____", null);
        }
    }
    /// <summary>
    /// Make sure the number mask shows even on blur
    /// </summary>
    [WebTestMethod]
    public void PhoneNumberMaskOnBlur()
    {
        NumberPage page = new NumberPage();
        Assert.AreEqual("___-___-____", page.PhoneNumberTextBox.GetAttributes().Value);
    }

    /// <summary>
    /// Make sure the number mask is in-place on focus.
    /// </summary>
    [WebTestMethod]
    public void PhoneNumberMaskOnFocus()
    {
        NumberPage page = new NumberPage();
        page.PhoneNumberTextBox.Focus();
        Assert.AreEqual("___-___-____", page.PhoneNumberTextBox.GetAttributes().Value);
    }

    /// <summary>
    /// Set the number with the mask and expect it to just work.
    /// </summary>
    [WebTestMethod]
    public void PhoneNumberSetWithMask()
    {
        NumberPage page = new NumberPage();
        page.PhoneNumberTextBox.SetText("100-000-0000");
        Assert.AreEqual("100-000-0000", page.PhoneNumberTextBox.GetAttributes().Value);
    }

    /// <summary>
    /// Enter just the number without the mask and expect the mask to fill in.
    /// </summary>
    [WebTestMethod]
    public void PhoneNumberSetWithoutMask()
    {
        NumberPage page = new NumberPage();
        page.PhoneNumberTextBox.SetText("1000000000");
        Assert.AreEqual("100-000-0000", page.PhoneNumberTextBox.GetAttributes().Value);
    }

    /// <summary>
    /// Entering invalid mask characters.
    /// </summary>
    /// <remarks>
    /// PASS: Manually it does not allow me to paste the mask but the test
    /// replaces the colons with dashes and just works. Not sure which one is better.
    /// </remarks>
    [WebTestMethod]
    public void PhoneNumberInvalidMask()
    {
        NumberPage page = new NumberPage();
        page.PhoneNumberTextBox.SetText("100:000:0000");
        Assert.AreEqual("100-000-0000", page.PhoneNumberTextBox.GetAttributes().Value);
    }

    /// <summary>
    /// Entering invalid characters should be disallowed and the fields 
    /// filled with zeroes.
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.All, "Expected: \"\", Actual: \"___-_12-3678\" and the zeroes are never entered on blur")]
    public void PhoneNumberInvalidAlphaNumericChars()
    {
        NumberPage page = new NumberPage();
        page.PhoneNumberTextBox.SetText("abc-123-a678");
        Assert.AreEqual("123-678-____", page.PhoneNumberTextBox.GetAttributes().Value);
        page.PhoneNumberTextBox.Blur();
        Assert.AreEqual("123-678-0000", page.PhoneNumberTextBox.GetAttributes().Value);
    }

    /// <summary>
    /// Make sure that the decimal points are in the right location
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.All, "Actual: \"12,30\"; when done manually in the browser, it depends  on where the cursor is. If the cursor is in the left most position the value is \"123.000,00\".")]
    public void CultureSensitiveDecimalPointHundredKNumber()
    {
        NumberPage page = new NumberPage();
        page.HundredKMaskTextBox.SetText("123", true);
        Assert.AreEqual("000.001,23", page.HundredKMaskTextBox.GetAttributes().Value ?? string.Empty);
    }

    /// <summary>
    /// Make sure that invalid chars are filtered out.
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.All, "Actual: \"134.567,80\" in FF; when done manually value is \"123.450,00\".")]
    public void InvalidCharactersHundredKNumber()
    {
        NumberPage page = new NumberPage();
        page.HundredKMaskTextBox.SetText("1a2b3c4d5e6f.7g8", true);
        Assert.AreEqual("123.456,78", page.HundredKMaskTextBox.GetAttributes().Value ?? string.Empty);
    }
}