// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.FilteredTextBoxExtender Model
    /// </summary>
    /// <TestComponent Name="FilteredTextBox">
    ///   <ToolkitType>AjaxControlToolkit.FilteredTextBoxExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.FilteredTextBox.FilteredTextBox_Common</CommonTestSuite>
    /// </TestComponent>
    public class FilteredTextBoxBehavior : Behavior<HtmlInputElement>
    {
        /// <summary>
        /// FilterType
        /// </summary>
        public BehaviorProperty<FilterTypes> FilterType
        {
            get { return _filterType; }
        }
        private BehaviorProperty<FilterTypes> _filterType;

        /// <summary>
        /// FilterMode
        /// </summary>
        public BehaviorProperty<FilterModes> FilterMode
        {
            get { return _filterMode; }
        }
        private BehaviorProperty<FilterModes> _filterMode;

        /// <summary>
        /// ValidChars
        /// </summary>
        public BehaviorProperty<string> ValidChars
        {
            get 
            {
                return _validChars; 
            }
        }
        private BehaviorProperty<string> _validChars;

        /// <summary>
        /// InvalidChars
        /// </summary>
        public BehaviorProperty<string> InvalidChars
        {
            get { return _invalidChars; }
        }
        private BehaviorProperty<string> _invalidChars;

        /// <summary>
        /// FilterInterval
        /// </summary>
        public BehaviorProperty<int> FilterInterval
        {
            get { return _filterInterval; }
        }
        private BehaviorProperty<int> _filterInterval;

        /// <summary>
        /// AjaxControlToolkit.FilteredTextBoxExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public FilteredTextBoxBehavior(HtmlInputElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _filterType = BehaviorProperty<FilterTypes>.CreateProperty(this, "FilterType");
            _filterMode = BehaviorProperty<FilterModes>.CreateProperty(this, "FilterMode");
            _validChars = BehaviorProperty<string>.CreateProperty(this, "ValidChars");
            _invalidChars = BehaviorProperty<string>.CreateProperty(this, "InvalidChars");
            _filterInterval = BehaviorProperty<int>.CreateProperty(this, "FilterInterval");
            _allowedChars = BehaviorProperty<string>.CreateCustomProperty(this, null, "{0}._getValidChars()", "throw 'AllowedChars property is readonly'");
        }

        /// <summary>
        /// Allowed chars computed on the basis of the FilterType property set
        /// </summary>
        public BehaviorProperty<string> AllowedChars
        {
            get
            {
                return this._allowedChars;
            }
        }

        private BehaviorProperty<string> _allowedChars;

        /// <summary>
        /// Append the specified text to the textbox and ensure that the resultant string 
        /// is filtered properly.
        /// </summary>
        /// <param name="textToAppendAndFilter">text to append</param>
        public void AppendTextAndAssertValid(string textToAppendAndFilter)
        {
            string textInTextBox = this.Element.GetAttributes().Value;
            this.SetTextAndAssertValid(textInTextBox + textToAppendAndFilter);
        }

        /// <summary>
        /// Set the text in the textbox to the specified value and ensure that 
        /// the filter is working correctly.
        /// </summary>
        /// <param name="textToFilter">text to filter</param>
        public void SetTextAndAssertValid(string textToFilter)
        {
            this.Element.SetText(textToFilter);
            string filteredText = this.Element.GetAttributes().Value;
            if (filteredText == null)
            {
                filteredText = String.Empty;
            }
            if ((this.FilterType.Value & FilterTypes.Custom) == FilterTypes.Custom)
            {
                if (this.FilterMode.Value == FilterModes.ValidChars)
                {

                    string netValidChars = this.ValidChars.Value + this.AllowedChars.Value;
                    this.AssertOnlyValidChars(filteredText, netValidChars.ToCharArray());
                }
                else if (this.FilterMode.Value == FilterModes.InvalidChars)
                {
                    this.AssertNoInvalidChars(filteredText, this.InvalidChars.Value.ToCharArray());
                }
            }
            else // non custom mode means that uppercase/lowercase/numeric has been set
            {
                this.AssertOnlyValidChars(filteredText, this.AllowedChars.Value.ToCharArray());
            }
        }

        private void AssertNoInvalidChars(string filteredText, char[] invalidChars)
        {
            int indexOfInvalidChar = filteredText.IndexOfAny(invalidChars);
            Assert.AreEqual(-1, indexOfInvalidChar);
        }

        private void AssertOnlyValidChars(string filteredText, char[] validChars)
        {
            // if only valid chars exist then splitting the string should result in a char array
            // of zero length
            string[] arraySplitByValidChars = filteredText.Split(validChars, StringSplitOptions.RemoveEmptyEntries);
            Assert.AreEqual(0, arraySplitByValidChars.Length);

        }
    }
}