// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Web.Script.Serialization;
using System.Web.UI.WebControls;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// Common test functionality
    /// </summary>
    public static class Common
    {
        #region ScrollIntoView
        /// <summary>
        /// Invoke a script that scrolls the element into view
        /// </summary>
        /// <param name="elementId">ID of the element to scroll into view</param>
        public static void ScrollIntoView(TestPage page, string elementId)
        {
            Assert.IsNotNull(page, "TestPage reference cannot be null!");
            Assert.StringIsNotNullOrEmpty(elementId, "elementId cannot be null or empty!");
            page.ExecuteScript(
                "(function() { var e = $get('" + elementId +
                "'); if (!e) { throw Error.argument('elementId', 'Failed to find element ID " +
                elementId + "'); }; e.scrollIntoView(); })()");
        }

        /// <summary>
        /// Invoke a script that scrolls the element into view
        /// </summary>
        /// <param name="element">Element to scroll into view</param>
        public static void ScrollIntoView(HtmlElement element)
        {
            Assert.IsNotNull(element, "element reference cannot be null!");
            ScrollIntoView(element.ParentPage, element.Id);
        }
        #endregion ScrollIntoView

        #region Wait
        /// <summary>
        /// Delegate to check for a condition
        /// </summary>
        /// <returns>Whether or not the condition has been satisfied</returns>
        public delegate bool WaitCondition();

        /// <summary>
        /// Wait until a condition has been satisfied and no assertions fail.
        /// </summary>
        /// <param name="page">Test page</param>
        /// <param name="seconds">Timeout length</param>
        /// <param name="condition">Condition</param>
        public static void Wait(TestPage page, int seconds, WaitCondition condition)
        {
            DateTime end = DateTime.Now.AddSeconds(seconds);
            WebTestingException lastFailure = null;
            while (end > DateTime.Now)
            {
                try
                {
                    if (condition())
                    {
                        return;
                    }
                }
                catch (WebTestingException ex)
                {
                    // Ignore any asserts that fire, but keep them for logging
                    lastFailure = ex;
                }

                Spin(page, 1);
            }
            throw new WebTestingException("Wait failed to satisfy condition", lastFailure);
        }

        /// <summary>
        /// Spin until a given number of seconds have elapsed
        /// </summary>
        /// <param name="page">Test page</param>
        /// <param name="seconds">seconds to spin</param>
        private static void Spin(TestPage page, int seconds)
        {
            page.ExecuteScript("(function() { window.___spin_timer = new Date(); return true; })()");
            page.WaitForScript("(new Date() - window.___spin_timer) < " + (seconds * 1000),
                seconds + 1);
        }
        #endregion Wait

        #region GetJson
        /// <summary>
        /// Get the value of a property from a deserialized JSON object
        /// </summary>
        /// <typeparam name="T">Type of the property</typeparam>
        /// <param name="obj">Deserialized JSON object</param>
        /// <param name="key">Name of the property</param>
        /// <returns>Value of the property</returns>
        public static T GetJson<T>(object obj, string key)
        {
            return GetJson<T>(obj, key, default(T));
        }

        /// <summary>
        /// Get the value of a property from a deserialized JSON object
        /// </summary>
        /// <typeparam name="T">Type of the property</typeparam>
        /// <param name="obj">Deserialized JSON object</param>
        /// <param name="key">Name of the property</param>
        /// <param name="defaultValue">Default value to use if the property is not found</param>
        /// <returns>Value of the property</returns>
        public static T GetJson<T>(object obj, string key, T defaultValue)
        {
            Assert.IsInstanceOfType(obj, typeof(IDictionary<string, object>));
            IDictionary<string, object> values = obj as IDictionary<string, object>;

            object value = null;
            return (values.TryGetValue(key, out value)) ?
                ConvertJsonValue<T>(value) :
                defaultValue;
        }

        /// <summary>
        /// Convert an object from a type returned by the JavaScriptSerializer
        /// into a value of the appropriate type.
        /// </summary>
        /// <typeparam name="T">Type to convert to</typeparam>
        /// <param name="value">Value to convert</param>
        /// <returns>Converted value</returns>
        public static T ConvertJsonValue<T>(object value)
        {
            // Hack: JavaScriptSerializer will serialize false as null
            if (typeof(T) == typeof(bool) && value == null)
            {
                value = false;
            }

            // Hack: JavaScriptSerializer will treat 1 as true and 0 as false
            if ((typeof(T) != typeof(bool)) && (value is bool))
            {
                value = (bool) value ? 1 : 0;
            }

            if (typeof(T) == typeof(bool?))
            {
                if (value != null)
                {
                    value = (bool?) Convert.ChangeType(value, typeof(bool));
                }
            }

            // Hack: Properly convert between numeric types
            if (!typeof(T).IsEnum && typeof(IConvertible).IsAssignableFrom(typeof(T)))
            {
                value = Convert.ChangeType(value, typeof(T));
            }

            // Ensure value types are non-null
            if (typeof(T).IsValueType && Nullable.GetUnderlyingType(typeof(T)) == null)
            {
                Assert.IsNotNull(value, "Value of type <{0}> cannot be null!", typeof(T).FullName);
            }
            
            // Ensure the value can be cast to the correct type
            T result = default(T);
            try { result = (T) value; }
            catch (InvalidCastException)
            {
                Assert.Fail("Value <{0}> should be of type <{1}> instead of <{2}>!",
                    value, typeof(T).FullName, value.GetType().FullName);
            }
            return result;
        }
        #endregion GetJson

        /// <summary>
        /// Move an element to specific coordinates
        /// </summary>
        /// <param name="element">Element to move</param>
        /// <param name="top">Right coordinate</param>
        /// <param name="left">Left coordinate</param>
        public static void Move(HtmlElement element, Unit top, Unit left)
        {
            Assert.IsNotNull(element);
            Assert.IsNotNull(element.ParentPage);
            Assert.StringIsNotNullOrEmpty(element.Id);

            element.ParentPage.ExecuteScript("(function() { " +
                "var e = $get('" + element.Id + "');" +
                "if (!e) { throw 'Failed to find element \"" + element.Id + "\"!'; }; " +
                "e.style.top = '" + top + "'; " +
                "e.style.left = '" + left + "'; " +
                "return true;})();");
        }

        /// <summary>
        /// Send a series of keystrokes to the element
        /// </summary>
        /// <param name="element">Element</param>
        /// <param name="text">String to be typed one character at a time</param>
        public static void TypeText(HtmlElement element, string text)
        {
            TypeText(element, text, true, true);
        }

        /// <summary>
        /// Send a series of keystrokes to the element
        /// </summary>
        /// <param name="element">Element</param>
        /// <param name="text">String to be typed one character at a time</param>
        /// <param name="focus">Focus the element before typing</param>
        /// <param name="blur">Blur the element after typing</param>
        public static void TypeText(HtmlElement element, string text, bool focus, bool blur)
        {
            Assert.IsNotNull(element);
            Assert.StringIsNotNullOrEmpty(text);

            if (focus)
            {
                element.Focus();
            }

            foreach (char ch in text)
            {
                HtmlKeyEvent @event = new HtmlKeyEvent("keydown");
                @event.KeyCode = ch;
                element.DispatchEvent(@event);

                // Firefox requires the KeyCode to be set for control characters
                // and IE will use either
                @event = new HtmlKeyEvent("keypress");
                if (!char.IsControl(ch))
                {
                    @event.CharCode = ch;
                }
                else
                {
                    @event.KeyCode = ch;
                }
                element.DispatchEvent(@event);

                @event = new HtmlKeyEvent("keyup");
                @event.KeyCode = ch;
                element.DispatchEvent(@event);
            }

            if (blur)
            {
                element.Blur();
            }
        }

        /// <summary>
        /// Gets the bounds of the specified HtmlElement
        /// </summary>
        /// <param name="element">element</param>
        /// <returns>Rectangle struct</returns>
        public static Rectangle GetBounds(HtmlElement element)
        {
            Assert.IsNotNull(element);
            Assert.StringIsNotNullOrEmpty(element.Id);
            IDictionary<string, object> dictionary = element.ParentPage.ExecuteScript(
                "(function() { "+
                    "var b = $common.getBounds($get('"+element.Id+"')); "+
                    "return { 'x': b.x, 'y': b.y, 'width': b.width, 'height': b.height }; "+
                "})()") as IDictionary<string, object>;
            Assert.IsNotNull(dictionary);
            return new Rectangle(Common.GetJson<int>(dictionary, "x"), Common.GetJson<int>(dictionary, "y"), Common.GetJson<int>(dictionary, "width"), Common.GetJson<int>(dictionary, "height"));
        }

        /// <summary>
        /// Gets the current style attribute of the specified HtmlElement
        /// </summary>
        /// <typeparam name="T">type of attribute value</typeparam>
        /// <param name="element">element</param>
        /// <param name="attribute">attribute</param>
        /// <returns>attribute value</returns>
        public static T GetCurrentStyle<T>(HtmlElement element, string attribute)
        {
            Assert.IsNotNull(element);
            Assert.StringIsNotNullOrEmpty(element.Id);
            Assert.StringIsNotNullOrEmpty(attribute);
            object result = element.ParentPage.ExecuteScript("$common.getCurrentStyle($get('" + element.Id + "'), '" + attribute + "')");
            return Common.ConvertJsonValue<T>(result);
        }
    }
}
