﻿// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.


// Dom elements for the currently executing test case and test command log entries
var currentTestCase = null;
var currentTestCaseTime = null;
var currentTestCommand = null;
var currentTestDetail = null;
var currentTestNode = null;

var syncLogs = false;
var progressAnimation = null;
var successColor = '#B0FFB0';
var failureColor = '#EFC4C1';

function getStatus(id) {
    /// <summary>
    /// Get an element from the status frame
    /// </summary>
    /// <param name="id" type="String" mayBeNull="false">Id of the element</param>
    return parent.testStatus.document.getElementById(id);
}

function getDriver(id) {
    /// <summary>
    /// Get an element from the driver frame
    /// </summary>
    /// <param name="id" type="String" mayBeNull="false">Id of the element</param>
    return document.getElementById(id);
}

function getResults(id) {
    /// <summary>
    /// Get an element from the results frame
    /// </summary>
    /// <param name="id" type="String" mayBeNull="false">Id of the element</param>
    return parent.testResults.document.getElementById(id);
}

function clickRunTests(btn) {
    /// <summary>
    /// Client-side handler invoked when clicking the Run Tests button
    /// </summary>
    /// <param name="btn" type="Sys.UI.DomElement" isDomElement="true">Run tests button</param>
    
    var start = getDriver('btnRun');
    if (start) {
        // Disable the status frame's button
        btn.disabled = true;
        btn.blur();
        
        // Disable other input controls on the status frame
        // and copy over their values
        var statusVerbosity = getStatus('cmbVerbosity');
        var driverVerbosity = getDriver('cmbVerbosity');
        if (statusVerbosity && driverVerbosity) {
            driverVerbosity.selectedIndex = statusVerbosity.selectedIndex;
            statusVerbosity.disabled = true;
        }
        
        var statusSkipFail = getStatus('chkSkipFail');
        var driverSkipFail = getDriver('chkSkipFail');
        if (statusSkipFail && driverSkipFail) {
            driverSkipFail.checked = statusSkipFail.checked;
            statusSkipFail.disabled = true;
        }
        
        var statusFilter = getStatus('chkFilter');
        var driverFilter = getDriver('chkFilter');
        if (statusFilter && driverFilter) {
            driverFilter.checked = statusFilter.checked;
            statusFilter.disabled = true;
        }
        
        // Click the button the driver frame
        start.click();        
    }
}

function toggleSyncLogs(checkbox) {
    /// <summary>
    /// Toggle whether or not we are currently syncing the logs
    /// </summary>
    /// <param name="checkbox" type="Sys.UI.DomElement" isDomElement="true" mayBeNull="false">
    /// Checkbox used for syncing logs
    /// </param>
    
    syncLogs = checkbox.checked;
    
    // Jump to the end of the logs if the user was in the
    // middle of a long step and just hit sync
    if (syncLogs) {
        if (currentTestDetail) {
            currentTestDetail.scrollIntoView();
        } else if (currentTestCommand) {
            currentTestCommand.scrollIntoView();
        } else if (currentTestCase) {
            currentTestCase.scrollIntoView();
        }
        
        if (currentTestNode) {
            currentTestNode.scrollIntoView();
        }
    }
}

TestExecutor.getLogContent = function() {
    /// <summary>
    /// Get the contents of the log to transfer to the server
    /// </summary>
    /// <returns type="String">Contents of the log</returns>
    
    return null;    // Return null if we're not logging anything
    
    // Example logging:
//    return '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">' +
//        '<html xmlns="http://www.w3.org/1999/xhtml" >' +
//        parent.testResults.document.documentElement.innerHTML +
//        '</html>';
};

TestExecutor.clearLog = function() {
    /// <summary>
    /// Clear the test log before another pass is started
    /// </summary>
    
    currentTestCase = null;
    currentTestCaseTime = null;
    currentTestCommand = null;
    currentTestDetail = null;
    currentTestNode = null;
    
    var log = getResults('results');
    if (log) {
        log.innerHTML = '&nbsp;';
    }
    
    var summary = getResults('summary');
    if (summary) {
        summary.innerHTML = '';
    }
    
    var lblTestId = getStatus('lblTestId');
    if (lblTestId) {
        lblTestId.innerHTML = '&nbsp;';
    }
    
    var progress = getStatus('progress');
    if (progress) {
        progress.style.width = '0px';
    }
    
    var progressValue  = getStatus('progressValue');
    if (progressValue) {
        progressValue.innerHTML = '0';
    }
    
    var lblFailures  = getStatus('lblFailures');
    if (lblFailures) {
        lblFailures.innerHTML = '0';
    }
    
    var lblCount  = getStatus('lblCount');
    if (lblCount) {
        lblCount.innerHTML = '0';
    }
    
    var lblTotal  = getStatus('lblTotal');
    if (lblTotal) {
        lblTotal.innerHTML = '0';
    }
    
    var lblTimer  = getStatus('lblTimer');
    if (lblTimer) {
        lblTimer.innerHTML = '';
    }
};
    
TestExecutor.logTestRunStarted = function(numberOfTests) {
    /// <summary>
    /// Log the start of a test run
    /// </summary>
    /// <param name="numberOfTests" type="Number">Number of tests in the rest run</param>
  
    // Prevent the default scripts from updating the TreeView with results
    TestExecutor.TestcaseExecutedCallback = null;
    
    var statusVerbosity = getStatus('cmbVerbosity');
    if (statusVerbosity) {
        statusVerbosity.disabled = true;
    }
        
    var statusSkipFail = getStatus('chkSkipFail');
    if (statusSkipFail) {
        statusSkipFail.disabled = true;
    }
    
    var statusFilter = getStatus('chkFilter');
    if (statusFilter) {
        statusFilter.disabled = true;
    }
    
    var statusSyncLogs = getStatus('chkSync');
    if (statusSyncLogs) {
        syncLogs = statusSyncLogs.checked;
    }
    
    var btn = getStatus("btnRun");
    if (btn) {
        btn.disabled = true;
    }
    
    var lblThread = getStatus('lblThread');
    if (lblThread) {
        lblThread.innerHTML = 'Running on thread ' + TestExecutor._threadId;
    }
    
    var lblTotal  = getStatus('lblTotal');
    if (lblTotal) {
        lblTotal.innerHTML = numberOfTests.toString();
    }
    
    var progress = getStatus('progress');
    if (progress) {
        progress.className = 'success';
    }
    
    if (!progressAnimation) {
        progressAnimation = new AjaxControlToolkit.Animation.LengthAnimation(progress, .2, 40, 'style', 'width', 0, 0, '%');
        var progressValue = getStatus('progressValue');
        progressAnimation.onStep = function(percentage) {
            var value = this.getAnimatedValue(percentage);
            this.setValue(value);
            progressValue.innerHTML = parseInt(value).toString();
        };
    }
    
    var status = getStatus('status');
    if (status) {
        status.style.visibility = 'visible';
    }
    
    var tree = getDriver('treeTestCases');
    if (tree) {
        var nodes = tree.getElementsByTagName('a');
        for (var i = 0; i < nodes.length; i++) {
            nodes[i]._onclick = nodes[i].onclick;
            nodes[i].onclick = function() { return false; };
            nodes[i].disabled = true;
        }
    }
    
    // Setup the timer
    TestExecutor.tickIntervalId = window.setInterval(timerTick, 1000);
    timerTick();
};

function timerTick() {
    /// <summary>
    /// Update the timer display every second
    /// </summary>

    var lblTimer  = getStatus('lblTimer');
    if (lblTimer) {
        lblTimer.innerHTML = TestExecutor.formatElapsedTime(TestExecutor._testRunStarted);
    }
}

TestExecutor.logTestRunFinished = function() {
    /// <summary>
    /// Log the end of a test run
    /// </summary>
    /// <param name="numberOfTests" type="Number">Number of tests in the rest run</param>
    
    // Remove the timer's tick and update the final time
    window.clearInterval(TestExecutor.tickIntervalId);
    timerTick();
    
    // Hide any necessary values
    var lblTestId = getStatus('lblTestId');
    if (lblTestId) {
        lblTestId.innerHTML = ' ';
    }
    
    var lblThread = getStatus('lblThread');
    if (lblThread) {
        lblThread.innerHTML = ' ';
    }
    
    var statusVerbosity = getStatus('cmbVerbosity');
    if (statusVerbosity) {
        statusVerbosity.disabled = false;
    }
        
    var statusSkipFail = getStatus('chkSkipFail');
    if (statusSkipFail) {
        statusSkipFail.disabled = false;
    }
    
    var statusFilter = getStatus('chkFilter');
    if (statusFilter) {
        statusFilter.disabled = false;
    }
    
    var btn = getStatus("btnRun");
    if (btn) {
        btn.disabled = false;
    }
    
    var tree = getDriver('treeTestCases');
    if (tree) {
        var nodes = tree.getElementsByTagName('a');
        for (var i = 0; i < nodes.length; i++) {
            nodes[i].onclick = nodes[i]._onclick;
        }
    }
    
    parent.testFrame.location.href = 'about:blank';
    
    var lblFailures  = getStatus('lblFailures');
    var lblTimer = getStatus('lblTimer');
    var summary = getResults('summary');
    if (lblFailures && lblTimer && summary) {
        summary.innerHTML = '<center>Test pass started at ' + TestExecutor._testRunStarted.format("T") + ' completed with <font color="red"><b>' + lblFailures.innerHTML + ' failure(s)</b></font> in <b>' + lblTimer.innerHTML + '</b></center>';
        if (syncLogs) {
            summary.scrollIntoView();
        }
    }
};

TestExecutor.logTestCaseStarted = function(testId) {
    /// <summary>
    /// Log the start of a test case
    /// </summary>
    /// <param name="testId" type="String">ID of the test case</param>
    
    var lblTestId = getStatus('lblTestId');
    if (lblTestId) {
        lblTestId.innerHTML = testId;
    }

    // Create the element for the new test case
    var log = getResults('results');
    if (log) {
        currentTestCase = parent.testResults.document.createElement('div');
        currentTestCase.id = testId;
        currentTestCase.className = 'testCase';
        log.appendChild(currentTestCase);
        
        var header = parent.testResults.document.createElement("div");
        header.className = 'testCaseHeader';
        currentTestCase.appendChild(header);
        
        var label = parent.testResults.document.createElement("div");
        label.innerHTML = testId;
        header.appendChild(label);
              
        currentTestCaseTime = parent.testResults.document.createElement("div")
        currentTestCaseTime.className = 'testCaseHeaderTime';
        currentTestCaseTime.innerHTML = 'Started at ' + TestExecutor._testCaseStarted.format("T");
        header.appendChild(currentTestCaseTime);
        
        if (syncLogs) {
            currentTestCase.scrollIntoView();
        }
        
        currentTestCommand = null;
        currentTestDetail = null;
    }
};

TestExecutor.logTestCaseFinished = function(testId, passed, errorMessage) {
    /// <summary>
    /// Log the end of a test case
    /// </summary>
    /// <param name="testId" type="String">ID of the test case</param>
    /// <param name="passed" type="Boolean">Whether the test passed or failed</param>
    /// <param name="errorMessage" type="String">Any error associated with the test</param>

    var progress = getStatus('progress');
    if (progress && !passed) {
        progress.className = 'failure';
    }
    
    var lblCount  = getStatus('lblCount');
    if (lblCount) {
        lblCount.innerHTML = TestExecutor._numberOfTestCasesRun.toString();
    }
    
    var lblFailures  = getStatus('lblFailures');
    if (lblFailures) {
        lblFailures.innerHTML = TestExecutor._numberOfTestCaseFailures.toString();
    }
    
    var percent = Math.floor(TestExecutor._numberOfTestCasesRun / TestExecutor._numberOfTestCases * 100);
    var progress = getStatus('progress');
    if (progress) {
        var start = progressAnimation.get_endValue();
        if (start < percent) {
            progressAnimation.set_startValue(start);
            progressAnimation.set_endValue(percent);
            progressAnimation.play();
        }
    }
    
    if (currentTestCase) {
        currentTestCase.childNodes[0].style.backgroundColor = passed ? successColor : failureColor;
        if (!passed)
        {
            currentTestCommand = null;
            TestExecutor.logTestDetail(errorMessage, true);
        }
    }
    
    if (currentTestCaseTime) {
        currentTestCaseTime.innerHTML = (passed ? 'Passed' : 'Failed') + ' in ' + TestExecutor.formatElapsedTime(TestExecutor._testCaseStarted);
    }
    
    var tree = getDriver('treeTestCases');
    if (tree) {
        var pattern = new RegExp(testId.replace(/\./, '(//|\\\\\\\\)'));
        var nodes = tree.getElementsByTagName('a');
        for (var i = 0; i < nodes.length; i++) {
            var node = nodes[i];
            if (pattern.test(node.href)) {
                node.style.backgroundColor = passed ? successColor : failureColor;
                node.href = "javascript:getResults('" + testId + "').scrollIntoView()";
                node.disabled = false;
                node.onclick = node._onclick;
                currentTestNode = node;
                
                if (syncLogs) {
                    currentTestNode.scrollIntoView();
                }
                break;
            }
        }
    }
};

TestExecutor.logTestCommand = function(command) {
    /// <summary>
    /// Log the execution of a test command
    /// </summary>
    /// <param name="command" type="Microsoft.Web.Testing.Engine.BrowserCommand">Command being executed</param>
       
    if (!currentTestCase) {
        return;
    }
    
    currentTestCommand = parent.testResults.document.createElement("div");
    currentTestCommand.className = 'testCommand';
    currentTestCase.appendChild(currentTestCommand);
    
    var commandHeader = parent.testResults.document.createElement("div");
    commandHeader.className = 'testCommandHeader';
    commandHeader.innerHTML = command.Description;
    currentTestCommand.appendChild(commandHeader);
    
    currentTestDetail = null;
    
    if (syncLogs) {
        currentTestCommand.scrollIntoView();
    }
};

TestExecutor.logTestDetail = function(detail, isHtml) {
    /// <summary>
    /// Log a detail associated with the command being executed
    /// </summary>
    /// <param name="detail" type="String">Detail of the command being executed</param>
    /// <param name="isHtml" type="Boolean">Whether the command is already formatted in HTML</param>
    
    currentTestDetail = parent.testResults.document.createElement("div");
    if (!isHtml) {
        detail = TestExecutor.htmlEncode(detail);
    }
    currentTestDetail.innerHTML = detail;
    
    // Place it in the deepest nested scope possible
    if (currentTestCommand) {
        currentTestDetail.className = 'testCommandDetail';
        currentTestCommand.appendChild(currentTestDetail);
    } else if (currentTestCase) {
        currentTestDetail.className = 'testCommand';
        currentTestCase.appendChild(currentTestDetail);
    } else if (log) {
        currentTestDetail.className = 'testCase';
        log.appendChild(currentTestDetail);
    }
    
    if (syncLogs) {
        currentTestDetail.scrollIntoView();
    }
};