// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// The ToolkitTestPage serves as a base class for all the page models
    /// in the client-side testing framework.  Each page model defines wrappers
    /// for all the client-side controls in a specific page.
    /// </summary>
    public abstract class ToolkitTestPage : TestPage
    {
        /// <summary>
        /// Path to the page being modeled
        /// </summary>
        public abstract string Path
        {
            get;
        }

        /// <summary>
        /// SynchronizationManager used to update all of the page's
        /// behaviors
        /// </summary>
        public SynchronizationManager SynchronizationManager
        {
            get { return _synchronizationManager; }
        }
        private SynchronizationManager _synchronizationManager;

        /// <summary>
        /// List of behaviors associated with the page (this will be
        /// populated automatically with the creation of Behavior&lt;T&gt;s).
        /// </summary>
        public IList<Behavior> Behaviors
        {
            get { return _behaviors; }
        }
        private List<Behavior> _behaviors;

        /// <summary>
        /// Default constructor that loads the page and all of its controls
        /// before the test cases start.
        /// </summary>
        public ToolkitTestPage()
            : this(NavigationVerification.AspNetAjax)
        {
        }

        /// <summary>
        /// Default constructor that loads the page and all of its controls
        /// before the test cases start.
        /// </summary>
        /// <param name="navigationVerification">NavigationVerification mode</param>
        /// <remarks>
        /// Although it's generally considered a bad practice, we are making
        /// virtual calls to the Page property and FindControls method in the
        /// constructor because it greatly simplifies usage of ToolkitTestPage.
        /// </remarks>
        public ToolkitTestPage(NavigationVerification navigationVerification)
            : base()
        {
            string path = Path;
            Assert.StringIsNotNullOrEmpty(path, "ToolkitTestPage requires a non-empty Path");
            Navigate(path, navigationVerification);

            _synchronizationManager = new SynchronizationManager(this);
            _behaviors = new List<Behavior>();
            FindControls();
        }

        /// <summary>
        /// The FindControls method is used to look up all of the controls
        /// used by the page model (and will be called automatically from the
        /// constructor
        /// </summary>
        public virtual void FindControls()
        {
        }

        /// <summary>
        /// Invalidate all behaviors passed as parameters or
        /// all of the behaviors on the page if nothing is passed.
        /// </summary>
        /// <param name="behaviors">behaviors to invalidate</param>
        public void Invalidate(params Behavior[] behaviors)
        {
            if ((behaviors == null) || (behaviors.Length == 0))
            {
                behaviors = _behaviors.ToArray();
            }
            foreach (Behavior behavior in _behaviors)
            {
                behavior.Invalidate();
            }
        }

        /// <summary>
        /// Synchronize all of the behaviors on this page
        /// </summary>
        public void Synchronize()
        {
            _synchronizationManager.Synchronize();
        }

        /// <summary>
        /// Call the .Invalidate method for the specified list of BehaviorProperties
        /// </summary>
        /// <param name="behaviorProperties">list of BehaviorProperties</param>
        public void InvalidateProperties(params BehaviorProperty[] behaviorProperties)
        {
            foreach (BehaviorProperty behaviorProperty in behaviorProperties)
            {
                behaviorProperty.Invalidate();
            }
        }

        /// <summary>
        /// Call the .Require methods for the specified list of BehaviorProperties
        /// </summary>
        /// <param name="behaviorProperties">list of BehaviorProperties</param>
        public void RequireProperties(params BehaviorProperty[] behaviorProperties)
        {
            foreach (BehaviorProperty behaviorProperty in behaviorProperties)
            {
                behaviorProperty.Require();
            }
        }

        /// <summary>
        /// Call the .Invalidate and .Require methods for the specified list of BehaviorProperties
        /// </summary>
        /// <param name="behaviorProperties">list of BehaviorProperties</param>
        public void InvalidateAndRequire(params BehaviorProperty[] behaviorProperties)
        {
            foreach (BehaviorProperty behaviorProperty in behaviorProperties)
            {
                behaviorProperty.Invalidate();
                behaviorProperty.Require();
            }
        }

        /// <summary>
        /// Register handlers for partial postback
        /// </summary>
        public void RegisterForPartialPostback()
        {
            object result = ExecuteScript(
                "(function() {" +
                    "window.asyncPostbackComplete = false;" +
                    "if (Sys && Sys.WebForms && Sys.WebForms.PageRequestManager) {" +
                        "var pageRequestManager = Sys.WebForms.PageRequestManager.getInstance();" +
                        "pageRequestManager.add_beginRequest(function() { window.asyncPostbackComplete = false; });" +
                        "pageRequestManager.add_endRequest(function() { window.asyncPostbackComplete = true; });" +
                        "return true;" +
                    "}" +
                    "return false;" +
                "})();");
            Assert.IsTrue(Common.ConvertJsonValue<bool>(result), "Failed to register Sys.WebForms.PageRequestManager handlers.");
        }

        /// <summary>
        /// Wait for partial postback to complete
        /// </summary>
        /// <remarks>
        /// RegisterForPartialPostback must have already been called
        /// </remarks>
        public void WaitForPartialPostback()
        {
            WaitForScript("window.asyncPostbackComplete", 30);
            ExecuteScript("window.asyncPostbackComplete = false;");
        }
    }
}