// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.CollapsiblePanelExtender Wrapper
    /// </summary>
    /// <TestComponent Name="CollapsiblePanel">
    ///   <ToolkitType>AjaxControlToolkit.CollapsiblePanelExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.CollapsiblePanel.CollapsiblePanel_Common</CommonTestSuite>
    /// </TestComponent>
    public class CollapsiblePanelBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// CollapseControl
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> CollapseControl
        {
            get { return _collapseControl; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _collapseControl;

        /// <summary>
        /// ExpandControl
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> ExpandControl
        {
            get { return _expandControl; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _expandControl;

        /// <summary>
        /// AutoCollapse
        /// </summary>
        public BehaviorProperty<bool> AutoCollapse
        {
            get { return _autoCollapse; }
        }
        private BehaviorProperty<bool> _autoCollapse;

        /// <summary>
        /// AutoExpand
        /// </summary>
        public BehaviorProperty<bool> AutoExpand
        {
            get { return _autoExpand; }
        }
        private BehaviorProperty<bool> _autoExpand;

        /// <summary>
        /// CollapsedSize
        /// </summary>
        public BehaviorProperty<int> CollapsedSize
        {
            get { return _collapsedSize; }
        }
        private BehaviorProperty<int> _collapsedSize;

        /// <summary>
        /// ExpandedSize
        /// </summary>
        public BehaviorProperty<int> ExpandedSize
        {
            get { return _expandedSize; }
        }
        private BehaviorProperty<int> _expandedSize;

        /// <summary>
        /// ScrollContents
        /// </summary>
        public BehaviorProperty<bool> ScrollContents
        {
            get { return _scrollContents; }
        }
        private BehaviorProperty<bool> _scrollContents;

        /// <summary>
        /// SuppressPostBack
        /// </summary>
        public BehaviorProperty<bool> SuppressPostBack
        {
            get { return _suppressPostBack; }
        }
        private BehaviorProperty<bool> _suppressPostBack;

        /// <summary>
        /// Collapsed
        /// </summary>
        public BehaviorProperty<bool> Collapsed
        {
            get { return _collapsed; }
        }
        private BehaviorProperty<bool> _collapsed;

        /// <summary>
        /// CollapsedText
        /// </summary>
        public BehaviorProperty<string> CollapsedText
        {
            get { return _collapsedText; }
        }
        private BehaviorProperty<string> _collapsedText;

        /// <summary>
        /// ExpandedText
        /// </summary>
        public BehaviorProperty<string> ExpandedText
        {
            get { return _expandedText; }
        }
        private BehaviorProperty<string> _expandedText;

        /// <summary>
        /// TextLabel
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> TextLabel
        {
            get { return _textLabel; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _textLabel;

        /// <summary>
        /// ExpandedImage
        /// </summary>
        public BehaviorProperty<string> ExpandedImage
        {
            get { return _expandedImage; }
        }
        private BehaviorProperty<string> _expandedImage;

        /// <summary>
        /// CollapsedImage
        /// </summary>
        public BehaviorProperty<string> CollapsedImage
        {
            get { return _collapsedImage; }
        }
        private BehaviorProperty<string> _collapsedImage;

        /// <summary>
        /// ImageControl
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> ImageControl
        {
            get { return _imageControl; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _imageControl;

        /// <summary>
        /// ExpandDirection
        /// </summary>
        public BehaviorProperty<CollapsiblePanelExpandDirection> ExpandDirection
        {
            get { return _expandDirection; }
        }
        private BehaviorProperty<CollapsiblePanelExpandDirection> _expandDirection;

        /// <summary>
        /// WrapperElement
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> WrapperElement
        {
            get { return _wrapperElement; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _wrapperElement;

        /// <summary>
        /// OffsetHeight
        /// </summary>
        public BehaviorProperty<int> OffsetHeight
        {
            get { return _offsetHeight; }
        }
        private BehaviorProperty<int> _offsetHeight;

        /// <summary>
        /// OffsetWidth
        /// </summary>
        public BehaviorProperty<int> OffsetWidth
        {
            get { return _offsetWidth; }
        }
        private BehaviorProperty<int> _offsetWidth;

        /// <summary>
        /// AjaxControlToolkit.CollapsiblePanelExtender Wrapper
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Wrapper</param>
        public CollapsiblePanelBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _collapseControl = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "CollapseControlID");
            _expandControl = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "ExpandControlID");
            _autoCollapse = BehaviorProperty<bool>.CreateProperty(this, "AutoCollapse");
            _autoExpand = BehaviorProperty<bool>.CreateProperty(this, "AutoExpand");
            _collapsedSize = BehaviorProperty<int>.CreateProperty(this, "CollapsedSize");
            _expandedSize = BehaviorProperty<int>.CreateProperty(this, "ExpandedSize");
            _scrollContents = BehaviorProperty<bool>.CreateProperty(this, "ScrollContents");
            _suppressPostBack = BehaviorProperty<bool>.CreateProperty(this, "SuppressPostBack");
            _collapsed = BehaviorProperty<bool>.CreateProperty(this, "Collapsed");
            _collapsedText = BehaviorProperty<string>.CreateProperty(this, "CollapsedText");
            _expandedText = BehaviorProperty<string>.CreateProperty(this, "ExpandedText");
            _textLabel = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "TextLabelID");
            _expandedImage = BehaviorProperty<string>.CreateProperty(this, "ExpandedImage");
            _collapsedImage = BehaviorProperty<string>.CreateProperty(this, "CollapsedImage");
            _imageControl = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "ImageControlID");
            _expandDirection = BehaviorProperty<CollapsiblePanelExpandDirection>.CreateProperty(this, "ExpandDirection");
            _wrapperElement = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", ReadStrategy.Demand, WriteStrategy.Immediate, "{0}._childDiv");
            _offsetHeight = BehaviorProperty<int>.CreateField(this, "offsetHeight", ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, "{0}._childDiv", "{0}.offsetHeight || 0", "{0}.offsetHeight = {1}");
            _offsetWidth = BehaviorProperty<int>.CreateField(this, "offsetWidth", ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, "{0}._childDiv", "{0}.offsetWidth || 0", "{0}.offsetWidth = {1}");
        }

        /// <summary>
        /// Get the position of the target
        /// </summary>
        public void GetTargetSize()
        {
            _offsetHeight.Invalidate();
            _offsetWidth.Invalidate();
            _offsetHeight.Require();
            _offsetWidth.Require();
        }

        /// <summary>
        /// Verify the panel is collapsed
        /// </summary>
        public void AssertCollapsed()
        {
            GetTargetSize();
            switch (_expandDirection.Value)
            {
                case AjaxControlToolkit.CollapsiblePanelExpandDirection.Vertical :
                    Assert.IsTrue(_offsetHeight.Value < 5,
                        "Height of target should be less than 5 (not {0})!", _offsetHeight.Value);
                    break;
                case CollapsiblePanelExpandDirection.Horizontal :
                    Assert.IsTrue(_offsetWidth.Value < 5,
                        "Width of target should be less than 5 (not {0})!", _offsetWidth.Value);
                    break;
            }
            Assert.AreEqual(_collapsedText.Value, _textLabel.Reference.GetAttributes().InnerText,
                "Text of the label should be \"{0}\" (not \"{1}\")!", _collapsedText.Value, _textLabel.Reference.GetAttributes().InnerText);
        }

        /// <summary>
        /// Verify the panel is expanded
        /// </summary>
        public void AssertExpanded()
        {
            GetTargetSize();
            switch (_expandDirection.Value)
            {
                case AjaxControlToolkit.CollapsiblePanelExpandDirection.Vertical:
                    Assert.IsTrue(_offsetHeight.Value >= 5,
                        "Height of target should be at least 5 (not {0})!", _offsetHeight.Value);
                    break;
                case CollapsiblePanelExpandDirection.Horizontal:
                    Assert.IsTrue(_offsetWidth.Value >= 5,
                        "Width of target should be at least 5 (not {0})!", _offsetWidth.Value);
                    break;
            }
            Assert.AreEqual(_expandedText.Value, _textLabel.Reference.GetAttributes().InnerText,
                "Text of the label should \"{0}\" (not \"{1}\")!", _expandedText.Value, _textLabel.Reference.GetAttributes().InnerText);
        }

        /// <summary>
        /// Verify the panel's state and collapse if necessary
        /// </summary>
        public void AssertAndCollapse()
        {
            if (Collapsed.Value)
            {
                AssertCollapsed();
            }
            else
            {
                AssertExpanded();
                Collapse();
            }
        }

        /// <summary>
        /// Verify the panel's state and expand if necessary
        /// </summary>
        public void AssertAndExpand()
        {
            if (Collapsed.Value)
            {
                AssertCollapsed();
                Expand();
            }
            else
            {
                AssertExpanded();
            }
        }

        /// <summary>
        /// Collapse the panel
        /// </summary>
        public void Collapse()
        {
            _collapseControl.Reference.Click();
            WaitForCollapse();
        }

        /// <summary>
        /// Expand the panel
        /// </summary>
        public void Expand()
        {
            _expandControl.Reference.Click();
            WaitForExpand();
        }

        /// <summary>
        /// Collapse the panel by calling the behavior's _doClose method
        /// </summary>
        public void BehaviorCollapse()
        {
            _suppressPostBack.Value = false;
            Page.ExecuteScript(string.Format("{0}._doClose()", BehaviorReferenceExpression));
            WaitForCollapse();
        }

        /// <summary>
        /// Expand the panel by calling the behavior's _doOpen method
        /// </summary>
        public void BehaviorExpand()
        {
            _suppressPostBack.Value = false;
            Page.ExecuteScript(string.Format("{0}._doOpen()", BehaviorReferenceExpression));
            WaitForExpand();
        }

        /// <summary>
        /// Wait until the panel has collapsed
        /// </summary>
        private void WaitForCollapse()
        {
            Common.Wait(Page, 3, delegate() { AssertCollapsed(); return true; });
        }

        /// <summary>
        /// Wait until the panel has expanded
        /// </summary>
        private void WaitForExpand()
        {
            Common.Wait(Page, 3, delegate() { AssertExpanded(); return true; });
        }
    }
}