// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.AutoCompleteExtender Wrapper
    /// </summary>
    /// <TestComponent Name="AutoComplete">
    ///   <ToolkitType>AjaxControlToolkit.AutoCompleteExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.AutoComplete.AutoComplete_Common</CommonTestSuite>
    /// </TestComponent>
    public class AutoCompleteBehavior : Behavior<HtmlInputElement>
    {
        /// <summary>
        /// MinimumPrefixLength
        /// </summary>
        public BehaviorProperty<int> MinimumPrefixLength
        {
            get { return _minimumPrefixLength; }
        }
        private BehaviorProperty<int> _minimumPrefixLength;

        /// <summary>
        /// CompletionInterval
        /// </summary>
        public BehaviorProperty<int> CompletionInterval
        {
            get { return _completionInterval; }
        }
        private BehaviorProperty<int> _completionInterval;

        /// <summary>
        /// CompletionSetCount
        /// </summary>
        public BehaviorProperty<int> CompletionSetCount
        {
            get { return _completionSetCount; }
        }
        private BehaviorProperty<int> _completionSetCount;

        /// <summary>
        /// CompletionListElementID
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> CompletionListElement
        {
            get { return _completionListElement; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _completionListElement;

        /// <summary>
        /// ServiceMethod
        /// </summary>
        public BehaviorProperty<string> ServiceMethod
        {
            get { return _serviceMethod; }
        }
        private BehaviorProperty<string> _serviceMethod;

        /// <summary>
        /// ServicePath
        /// </summary>
        public BehaviorProperty<string> ServicePath
        {
            get { return _servicePath; }
        }
        private BehaviorProperty<string> _servicePath;

        /// <summary>
        /// ContextKey
        /// </summary>
        public BehaviorProperty<string> ContextKey
        {
            get { return _contextKey; }
        }
        private BehaviorProperty<string> _contextKey;

        /// <summary>
        /// UseContextKey
        /// </summary>
        public BehaviorProperty<bool> UseContextKey
        {
            get { return _useContextKey; }
        }
        private BehaviorProperty<bool> _useContextKey;

        /// <summary>
        /// CompletionListCssClass
        /// </summary>
        public BehaviorProperty<string> CompletionListCssClass
        {
            get { return _completionListCssClass; }
        }
        private BehaviorProperty<string> _completionListCssClass;

        /// <summary>
        /// CompletionListItemCssClass
        /// </summary>
        public BehaviorProperty<string> CompletionListItemCssClass
        {
            get { return _completionListItemCssClass; }
        }
        private BehaviorProperty<string> _completionListItemCssClass;

        /// <summary>
        /// CompletionListHighlightedItemCssClass
        /// </summary>
        public BehaviorProperty<string> CompletionListHighlightedItemCssClass
        {
            get { return _completionListHighlightedItemCssClass; }
        }
        private BehaviorProperty<string> _completionListHighlightedItemCssClass;

        /// <summary>
        /// EnableCaching
        /// </summary>
        public BehaviorProperty<bool> EnableCaching
        {
            get { return _enableCaching; }
        }
        private BehaviorProperty<bool> _enableCaching;

        /// <summary>
        /// DelimiterCharacters
        /// </summary>
        public BehaviorProperty<string> DelimiterCharacters
        {
            get { return _delimiterCharacters; }
        }
        private BehaviorProperty<string> _delimiterCharacters;

        /// <summary>
        /// FirstRowSelected
        /// </summary>
        public BehaviorProperty<bool> FirstRowSelected
        {
            get { return _firstRowSelected; }
        }
        private BehaviorProperty<bool> _firstRowSelected;

        /// <summary>
        /// PopupBehavior
        /// </summary>
        public PopupBehavior PopupBehavior
        {
            get 
            {
                if (_popupBehavior == null)
                {
                    _popupBehavior = new PopupBehavior(
                        _completionListElement.Reference,
                        BehaviorID + "PopupBehavior",
                        string.Format("{0}._popupBehavior", this.BehaviorReferenceExpression),
                        Page);
                }
                return _popupBehavior; 
            }
        }
        private PopupBehavior _popupBehavior;

        ///// <summary>
        ///// OnClientPopulating
        ///// </summary>
        //public BehaviorProperty<string> OnClientPopulating
        //{
        //    get { return _onClientPopulating; }
        //}
        //private BehaviorProperty<string> _onClientPopulating;

        ///// <summary>
        ///// OnClientPopulated
        ///// </summary>
        //public BehaviorProperty<string> OnClientPopulated
        //{
        //    get { return _onClientPopulated; }
        //}
        //private BehaviorProperty<string> _onClientPopulated;

        ///// <summary>
        ///// OnClientShowing
        ///// </summary>
        //public BehaviorProperty<string> OnClientShowing
        //{
        //    get { return _onClientShowing; }
        //}
        //private BehaviorProperty<string> _onClientShowing;

        ///// <summary>
        ///// OnClientShown
        ///// </summary>
        //public BehaviorProperty<string> OnClientShown
        //{
        //    get { return _onClientShown; }
        //}
        //private BehaviorProperty<string> _onClientShown;

        ///// <summary>
        ///// OnClientHiding
        ///// </summary>
        //public BehaviorProperty<string> OnClientHiding
        //{
        //    get { return _onClientHiding; }
        //}
        //private BehaviorProperty<string> _onClientHiding;

        ///// <summary>
        ///// OnClientHidden
        ///// </summary>
        //public BehaviorProperty<string> OnClientHidden
        //{
        //    get { return _onClientHidden; }
        //}
        //private BehaviorProperty<string> _onClientHidden;

        ///// <summary>
        ///// OnClientItemSelected
        ///// </summary>
        //public BehaviorProperty<string> OnClientItemSelected
        //{
        //    get { return _onClientItemSelected; }
        //}
        //private BehaviorProperty<string> _onClientItemSelected;

        ///// <summary>
        ///// OnClientItemOver
        ///// </summary>
        //public BehaviorProperty<string> OnClientItemOver
        //{
        //    get { return _onClientItemOver; }
        //}
        //private BehaviorProperty<string> _onClientItemOver;

        ///// <summary>
        ///// OnClientItemOut
        ///// </summary>
        //public BehaviorProperty<string> OnClientItemOut
        //{
        //    get { return _onClientItemOut; }
        //}
        //private BehaviorProperty<string> _onClientItemOut;

        /// <summary>
        /// AjaxControlToolkit.AutoCompleteExtender Wrapper
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Wrapper</param>
        public AutoCompleteBehavior(HtmlInputElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _minimumPrefixLength = BehaviorProperty<int>.CreateProperty(this, "minimumPrefixLength");
            _completionInterval = BehaviorProperty<int>.CreateProperty(this, "completionInterval");
            _completionSetCount = BehaviorProperty<int>.CreateProperty(this, "completionSetCount");
            _completionListElement = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}._completionListElement");
            _serviceMethod = BehaviorProperty<string>.CreateProperty(this, "serviceMethod");
            _servicePath = BehaviorProperty<string>.CreateProperty(this, "servicePath");
            _contextKey = BehaviorProperty<string>.CreateProperty(this, "contextKey");
            _useContextKey = BehaviorProperty<bool>.CreateProperty(this, "useContextKey");
            _completionListCssClass = BehaviorProperty<string>.CreateProperty(this, "completionListCssClass");
            _completionListItemCssClass = BehaviorProperty<string>.CreateProperty(this, "completionListItemCssClass");
            _completionListHighlightedItemCssClass = BehaviorProperty<string>.CreateProperty(this, "highlightedItemCssClass");
            _enableCaching = BehaviorProperty<bool>.CreateProperty(this, "enableCaching");
            _delimiterCharacters = BehaviorProperty<string>.CreateProperty(this, "delimiterCharacters");
            _firstRowSelected = BehaviorProperty<bool>.CreateProperty(this, "firstRowSelected");
        }

        /// <summary>
        /// Sets the specified text in the textbox and waits for the flyout to show up.
        /// </summary>
        /// <param name="text">String to set in the textbox</param>
        public void SetTextWaitForFlyout(string text)
        {
            this._actualCompletionSetCount = 0;
            this.PopupBehavior.Shown.Register();
            this.Element.Focus();
            this.Element.SetText(text, false);
            this.Element.DispatchEvent(new HtmlKeyEvent("keydown"));
            this.PopupBehavior.Shown.Wait(5, false);
            Assert.IsTrue(this.ActualCompletionSetCount > 0);
        }

        /// <summary>
        /// Gets the suggestions in the flyout
        /// </summary>
        /// <returns>String array that contains the suggestions.</returns>
        public string[] GetSuggestions()
        {   
            //for(int i = 0 ; i <= 
            //if (index > 0 && index < this.ActualCompletionSetCount)
            //{
            //    return this.CompletionListElement.Reference.ChildElements[index];
            //}
            return null;
        }

        /// <summary>
        /// Returns the number of suggestions in the flyout
        /// </summary>
        public int ActualCompletionSetCount
        {
            get
            {
                if (this._actualCompletionSetCount <= 0)
                {
                    this.CompletionListElement.Reference.ChildElements.Refresh();
                    this._actualCompletionSetCount = this.CompletionListElement.Reference.ChildElements.Count;
                }
                return this._actualCompletionSetCount;
            }

        }

        private int _actualCompletionSetCount;

        /// <summary>
        /// Returns the word to complete in the textbox
        /// </summary>
        /// <returns>The word to complete.</returns>
        public string GetCurrentWordToComplete()
        {
            return (string)this.Page.ExecuteScript(string.Format("{0}._currentCompletionWord()", BehaviorReferenceExpression));
        }

        /// <summary>
        /// Select the option at the specified index.
        /// </summary>
        /// <param name="index"></param>
        public void ChooseOption(int index)
        {
            this.ValidateIndex(index);
            this.CompletionListElement.Reference.ChildElements[index].DispatchEvent(new HtmlMouseEvent("mousedown"));
        }

        /// <summary>
        /// Get the autocomplete suggestion/option
        /// </summary>
        /// <param name="index">number at which option is located in the collection</param>
        /// <returns>the suggestion text</returns>
        public string GetOption(int index)
        {
            this.ValidateIndex(index);
            HtmlElement optionElem = this.CompletionListElement.Reference.ChildElements[index];
            return optionElem.GetAttributes().InnerText.Trim();
        }

        public string TextBoxValue
        {
            get
            {
                return this.Element.GetAttributes().Value;
            }
        }

        /// <summary>
        /// Make sure that the index is valid
        /// </summary>
        /// <param name="index">index</param>
        private void ValidateIndex(int index)
        {
            if (index < 0 || index >= this.ActualCompletionSetCount)
            {
                throw new IndexOutOfRangeException(string.Format("Invalid option index: {0}", index));
            }
        }
    }
}