// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.Web.UI.WebControls;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.AlwaysVisibleControlExtender Wrapper
    /// </summary>
    /// <TestComponent Name="AlwaysVisibleControl">
    ///   <ToolkitType>AjaxControlToolkit.AlwaysVisibleControlExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.AlwaysVisibleControl.AlwaysVisibleControl_Common</CommonTestSuite>
    /// </TestComponent>
    public class AlwaysVisibleControlBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// HorizontalOffset
        /// </summary>
        public BehaviorProperty<int> HorizontalOffset
        {
            get { return _horizontalOffset; }
        }
        private BehaviorProperty<int> _horizontalOffset;

        /// <summary>
        /// HorizontalSide
        /// </summary>
        public BehaviorProperty<HorizontalSide> HorizontalSide
        {
            get { return _horizontalSide; }
        }
        private BehaviorProperty<HorizontalSide> _horizontalSide;

        /// <summary>
        /// VerticalOffset
        /// </summary>
        public BehaviorProperty<int> VerticalOffset
        {
            get { return _verticalOffset; }
        }
        private BehaviorProperty<int> _verticalOffset;

        /// <summary>
        /// VerticalSide
        /// </summary>
        public BehaviorProperty<VerticalSide> VerticalSide
        {
            get { return _verticalSide; }
        }
        private BehaviorProperty<VerticalSide> _verticalSide;

        /// <summary>
        /// ScrollEffectDuration
        /// </summary>
        public BehaviorProperty<float> ScrollEffectDuration
        {
            get { return _scrollEffectDuration; }
        }
        private BehaviorProperty<float> _scrollEffectDuration;

        /// <summary>
        /// RequiresAnimation
        /// </summary>
        public BehaviorProperty<bool> RequiresAnimation
        {
            get { return _requiresAnimation; }
        }
        private BehaviorProperty<bool> _requiresAnimation;

        /// <summary>
        /// OffsetTop
        /// </summary>
        public BehaviorProperty<int> OffsetTop
        {
            get { return _offsetTop; }
        }
        private BehaviorProperty<int> _offsetTop;

        /// <summary>
        /// OffsetLeft
        /// </summary>
        public BehaviorProperty<int> OffsetLeft
        {
            get { return _offsetLeft; }
        }
        private BehaviorProperty<int> _offsetLeft;

        /// <summary>
        /// Repositioning event
        /// </summary>
        public BehaviorEvent Repositioning
        {
            get { return _repositioning; }
        }
        private BehaviorEvent _repositioning;

        /// <summary>
        /// Repositioned event
        /// </summary>
        public BehaviorEvent Repositioned
        {
            get { return _repositioned; }
        }
        private BehaviorEvent _repositioned;

        /// <summary>
        /// AjaxControlToolkit.AlwaysVisibleControlExtender Wrapper
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Wrapper</param>
        public AlwaysVisibleControlBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _horizontalOffset = BehaviorProperty<int>.CreateProperty(this, "HorizontalOffset", ReadStrategy.DemandAndInitialize, WriteStrategy.NextSynchronization);
            _horizontalSide = BehaviorProperty<HorizontalSide>.CreateProperty(this, "HorizontalSide", ReadStrategy.DemandAndInitialize, WriteStrategy.NextSynchronization);
            _verticalOffset = BehaviorProperty<int>.CreateProperty(this, "VerticalOffset", ReadStrategy.DemandAndInitialize, WriteStrategy.NextSynchronization);
            _verticalSide = BehaviorProperty<VerticalSide>.CreateProperty(this, "VerticalSide", ReadStrategy.DemandAndInitialize, WriteStrategy.NextSynchronization);
            _scrollEffectDuration = BehaviorProperty<float>.CreateProperty(this, "ScrollEffectDuration", ReadStrategy.Demand, WriteStrategy.NextSynchronization);
            _requiresAnimation = BehaviorProperty<bool>.CreateField(this, "_animate", ReadStrategy.Demand, WriteStrategy.Immediate);
            _offsetTop = BehaviorProperty<int>.CreateField(this, "offsetTop", ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, "{0}._element");
            _offsetLeft = BehaviorProperty<int>.CreateField(this, "offsetLeft", ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, "{0}._element");
            _repositioning = BehaviorEvent<object>.CreateEvent(this, "repositioning");
            _repositioned = BehaviorEvent<object>.CreateEvent(this, "repositioned");
        }

        /// <summary>
        /// Get the position of the target
        /// </summary>
        public void GetTargetPosition()
        {
            _offsetTop.Invalidate();
            _offsetLeft.Invalidate();
            _offsetTop.Require();
            _offsetLeft.Require();
        }

        /// <summary>
        /// Verify the position of the target
        /// </summary>
        /// <param name="expectedTop">Expected top position</param>
        /// <param name="expectedLeft">Expected left position</param>
        public void AssertPosition(Unit expectedTop, Unit expectedLeft)
        {
            GetTargetPosition();
            Unit actualTop = _offsetTop.Value;
            Unit actualLeft = _offsetLeft.Value;
            Assert.AreEqual(expectedTop, actualTop,
                "Top of element \"{0}\" should be at position {1} (not {2})!", Element.Id, expectedTop, actualTop);
            Assert.AreEqual(expectedLeft, actualLeft,
                "Left of element \"{0}\" should be at position {1} (not {2})!", Element.Id, expectedLeft, actualLeft);
        }

        /// <summary>
        /// Verify the target has moved vertically from its initial position
        /// </summary>
        /// <param name="originalTop">Original top position</param>
        public void AssertHasMovedVertically(Unit originalTop)
        {
            GetTargetPosition();
            Assert.AreNotEqual(originalTop, _offsetTop.Value,
                "Top of element \"{0}\" should have moved from {1}!", Element.Id, originalTop);
        }

        /// <summary>
        /// Verify the target has moved horizontally from its initial position
        /// </summary>
        /// <param name="originalTop">Original left position</param>
        public void AssertHasMovedHorizontally(Unit originalLeft)
        {
            GetTargetPosition();
            Assert.AreNotEqual(originalLeft, _offsetLeft.Value,
                "Left of element \"{0}\" should have moved from {1}!", Element.Id, originalLeft);
        }
    }
}