﻿<%@ Application Language="C#" %>
<%@ Import Namespace="System.Collections.Generic" %>
<%@ Import Namespace="System.Text.RegularExpressions" %>
<script runat="server">
    // (c) Copyright Microsoft Corporation.
    // This source is subject to the Microsoft Public License.
    // See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
    // All other rights reserved.
    
    
    /// <summary>
    /// Regular expression used to rewrite tag urls
    /// </summary>
    /// <remarks>
    /// Group 0: Whole match
    /// Group 3: TagName
    /// Group 5: /Run
    /// Group 6: /Filter
    /// Group 7: /SkipFail
    /// Group 8: /Verbose
    /// Group 9: /Concise
    /// Group 10: /SyncLogs
    /// Group 11: Other content that needs to be rebased
    /// </remarks>
    private static Regex TagRewritingExpression = new Regex(@"^.*\/tag(\/(?!(testdriver|run|filter|skipfail|verbose|concise|synclogs))([^\/\?=]+))?((\/run)|(\/filter)|(\/skipfail)|(\/verbose)|(\/concise)|(\/synclogs))*(\/.*)?$", RegexOptions.IgnoreCase | RegexOptions.Compiled);
    
    /// <summary>
    /// Rewrite any urls with tags or auto-run instructions
    /// </summary>
    /// <param name="sender">Application</param>
    /// <param name="e">EventArgs</param>
    protected void Application_BeginRequest(object sender, EventArgs e)
    {
        // Check if the url should be rewritten
        Match match = TagRewritingExpression.Match(Request.Url.ToString());
        if (!match.Success || match.Groups.Count != 12)
        {
            return;
        }

        // Fix any rebased content
        string desiredContent = match.Groups[11].Value;
        if (!string.IsNullOrEmpty(desiredContent) && (desiredContent != "/"))
        {
            // Redirect to the real url
            Context.RewritePath('~' + desiredContent, false);
            return;
        }
        
        // Get the tag and whether we should auto-run
        string tag = match.Groups[3].Value;
        bool run = match.Groups[5].Length > 0;
        bool filter = match.Groups[6].Length > 0;
        bool skipFail = match.Groups[7].Length > 0;
        bool verbose = match.Groups[8].Length > 0;
        bool concise = match.Groups[9].Length > 0;
        bool syncLogs = match.Groups[10].Length > 0;

        // Create the real url
        StringBuilder url = new StringBuilder("~/Default.aspx?");
        List<string> directives = new List<string>();
        if (!string.IsNullOrEmpty(tag))
        {
            directives.Add(string.Format("tag={0}", tag));
        }
        if (run)
        {
            directives.Add("run=true");
        }
        if (filter)
        {
            directives.Add("filter=true");
        }
        if (skipFail)
        {
            directives.Add("skipfail=true");
        }
        if (verbose)
        {
            directives.Add("verbose=true");
        }
        if (concise)
        {
            directives.Add("concise=true");
        }
        if (syncLogs)
        {
            directives.Add("synclogs=true");
        }
        url.Append(string.Join("&", directives.ToArray()));

        // Redirect to the real url
        Context.RewritePath(url.ToString(), true);
    }
</script>