// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;

namespace AjaxControlToolkit.Testing.Client.PopupControl
{
    /// <summary>
    /// Common PopupControl tests
    /// </summary>
    /// <TestSuite>
    ///   <ComponentType>AjaxControlToolkit.Testing.Client.PopupControlBehavior</ComponentType>
    /// </TestSuite>
    [WebTestTag("PopupControl")]
    [WebTestTag("Popup")]
    public abstract class PopupControl_Common
    {
        public const string CommitText = "Commit";

        public abstract IEnumerable<PopupControlScenario> PopupControlScenarios
        {
            get;
        }

        [WebTestMethod]
        public void Common_InitialState()
        {
            foreach (PopupControlScenario popupControlScenario in PopupControlScenarios)
            {
                Assert.StringIsNullOrEmpty(popupControlScenario.Element.GetAttributes().Value);
                Assert.AreEqual("hidden", Common.GetCurrentStyle<string>(popupControlScenario.Popup, "visibility"));
                popupControlScenario.Element.Focus();
                Assert.AreNotEqual("hidden", Common.GetCurrentStyle<string>(popupControlScenario.Popup, "visibility"));
                popupControlScenario.Element.ParentElement.Click();
                Assert.AreEqual("hidden", Common.GetCurrentStyle<string>(popupControlScenario.Popup, "visibility"));
                popupControlScenario.Element.Click();
                Assert.AreNotEqual("hidden", Common.GetCurrentStyle<string>(popupControlScenario.Popup, "visibility"));
                popupControlScenario.Element.ParentElement.Click();
                Assert.AreEqual("hidden", Common.GetCurrentStyle<string>(popupControlScenario.Popup, "visibility"));
                Assert.StringIsNullOrEmpty(popupControlScenario.Element.GetAttributes().Value);
            }
        }

        [WebTestMethod]
        public virtual void Common_Commit()
        {
            foreach (PopupControlScenario popupControlScenario in PopupControlScenarios)
            {
                if ((null != popupControlScenario.Commit) && (null != popupControlScenario.Clear))
                {
                    AssertPopupClickText(popupControlScenario, popupControlScenario.Commit, CommitText);
                    AssertPopupClickText(popupControlScenario, popupControlScenario.Clear, null);
                }
            }
        }

        [WebTestMethod]
        public void Common_Cancel()
        {
            foreach (PopupControlScenario popupControlScenario in PopupControlScenarios)
            {
                if ((null != popupControlScenario.Commit) && (null != popupControlScenario.Cancel))
                {
                    AssertPopupClickText(popupControlScenario, popupControlScenario.Cancel, null);
                    AssertPopupClickText(popupControlScenario, popupControlScenario.Commit, CommitText);
                    AssertPopupClickText(popupControlScenario, popupControlScenario.Cancel, CommitText);
                }
            }
        }

        [WebTestMethod]
        public void Common_MultipleCommitCancel()
        {
            foreach (PopupControlScenario popupControlScenario in PopupControlScenarios)
            {
                if ((null != popupControlScenario.CommitCancel) && (null != popupControlScenario.CancelCommit))
                {
                    AssertPopupClickText(popupControlScenario, popupControlScenario.CommitCancel, null);
                    AssertPopupClickText(popupControlScenario, popupControlScenario.CancelCommit, CommitText);
                }
            }
        }

        protected void AssertPopupClickText(PopupControlScenario popupControlScenario, HtmlInputElement button, string text)
        {
            popupControlScenario.Element.Focus();
            popupControlScenario.PopupControlBehavior.PopupBehavior.Hidden.Register();
            button.Click();
            WaitForAsyncPostback(popupControlScenario.PopupControlBehavior);
            // Note: Will fail if the button is not in an UpdatePanel (as it should be for PopupControl)
            popupControlScenario.PopupControlBehavior.PopupBehavior.Hidden.Wait(5);
            if (string.IsNullOrEmpty(text))
            {
                Assert.StringIsNullOrEmpty(popupControlScenario.Element.GetAttributes().Value);
            }
            else
            {
                Assert.AreEqual(text, popupControlScenario.Element.GetAttributes().Value);
            }
        }

        private void WaitForAsyncPostback(PopupControlBehavior behavior)
        {
            // Give the sync/async postback time to start, then wait until the page is loaded and initialized
            int timeRequiredToStartPostback = 100;
            System.Threading.Thread.Sleep(timeRequiredToStartPostback);
            behavior.Page.WaitForScript(string.Format("(null != $find) && (null != {0})", behavior.BehaviorReferenceExpression), 3);
        }
    }
}
