// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.PopupExtender Model
    /// </summary>
    /// <TestComponent Name="Popup">
    ///   <ToolkitType>AjaxControlToolkit.PopupExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.Popup.Popup_Common</CommonTestSuite>
    /// </TestComponent>
    public class PopupBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// ParentElement
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> ParentElement
        {
            get { return _parentElement; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _parentElement;

        /// <summary>
        /// X
        /// </summary>
        public BehaviorProperty<int> X
        {
            get { return _x; }
        }
        private BehaviorProperty<int> _x;

        /// <summary>
        /// Y
        /// </summary>
        public BehaviorProperty<int> Y
        {
            get { return _y; }
        }
        private BehaviorProperty<int> _y;

        /// <summary>
        /// PositioningMode
        /// </summary>
        public BehaviorProperty<PositioningMode> PositioningMode
        {
            get { return _positioningMode; }
        }
        private BehaviorProperty<PositioningMode> _positioningMode;

        ///// <summary>
        ///// Reparent
        ///// </summary>
        //public BehaviorProperty<bool> Reparent
        //{
        //    get { return _reparent; }
        //}
        //private BehaviorProperty<bool> _reparent;

        /// <summary>
        /// OnShow
        /// </summary>
        public GenericAnimationBehavior OnShow
        {
            get { return _onShow; }
        }
        private GenericAnimationBehavior _onShow;

        /// <summary>
        /// OnHide
        /// </summary>
        public GenericAnimationBehavior OnHide
        {
            get { return _onHide; }
        }
        private GenericAnimationBehavior _onHide;

        /// <summary>
        /// Showing
        /// </summary>
        public BehaviorEvent Showing
        {
            get { return _showing; }
        }
        private BehaviorEvent _showing;

        /// <summary>
        /// Shown
        /// </summary>
        public BehaviorEvent Shown
        {
            get { return _shown; }
        }
        private BehaviorEvent _shown;

        /// <summary>
        /// Hiding
        /// </summary>
        public BehaviorEvent Hiding
        {
            get { return _hiding; }
        }
        private BehaviorEvent _hiding;

        /// <summary>
        /// Hidden
        /// </summary>
        public BehaviorEvent Hidden
        {
            get { return _hidden; }
        }
        private BehaviorEvent _hidden;

        /// <summary>
        /// AjaxControlToolkit.PopupExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public PopupBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : this (element, behaviorID, null, page)
        { }

        /// <summary>
        /// AjaxControlToolkit.PopupExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="behaviorReferenceExpression">JavaScript expression used to reference the behavior</param>
        /// <param name="page">Page Model</param>
        public PopupBehavior(HtmlElement element, string behaviorID, string behaviorReferenceExpression, ToolkitTestPage page)
            : base (element, behaviorID, behaviorReferenceExpression, page)
        {
            
            _parentElement = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}._parentElement");
            _x = BehaviorProperty<int>.CreateProperty(this, "x");
            _y = BehaviorProperty<int>.CreateProperty(this, "y");
            _positioningMode = BehaviorProperty<PositioningMode>.CreateProperty(this, "positioningMode");
            // commenting this out since this does not exist on the client side
            // _reparent = BehaviorProperty<bool>.CreateProperty(this, "reparent");
            _onShow = new GenericAnimationBehavior(
                        element,
                        behaviorID + "_onShow",
                        string.Format("{0}._onShow", BehaviorReferenceExpression),
                        page);
            _onHide = new GenericAnimationBehavior(
                        element,
                        behaviorID + "_onHide",
                        string.Format("{0}._onHide", BehaviorReferenceExpression),
                        page);
            _showing = BehaviorEvent<object>.CreateEvent(this, "showing");
            _shown = BehaviorEvent<object>.CreateEvent(this, "shown");
            _hiding = BehaviorEvent<object>.CreateEvent(this, "hiding");
            _hidden = BehaviorEvent<object>.CreateEvent(this, "hidden");
        }

        public void Show()
        {
            this.Page.ExecuteScript(string.Format("{0}.show()", BehaviorReferenceExpression));
            Assert.IsFalse(this.IsHidden);
        }

        public void Hide()
        {
            this.Page.ExecuteScript(string.Format("{0}.hide()", BehaviorReferenceExpression));
            Assert.IsTrue(this.IsHidden);
        }

        public bool IsHidden
        {
            get
            {
                HtmlStyle style = this.Element.GetAttributes().Style;
                return ((style.Display == Display.None) || (style.Visibility == Visibility.Hidden));
            }
        }
    }
}