// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit.Testing.Client;
using AjaxControlToolkit.Testing.Client.MaskedEdit;

/// <summary>
/// Ensure that masked edit extender works fine if the mask type is set to Date
/// </summary>
/// <TestSuite>
///   <Path>MaskedEdit\Date.aspx</Path>
///   <ComponentType>AjaxControlToolkit.Testing.Client.MaskedEditBehavior</ComponentType>
///   <WrapperType>AjaxControlToolkit.Testing.Client.MaskedEdit.DatePage</WrapperType>
/// </TestSuite>
[WebTestClass]
[WebTestTag("MaskedEdit.Date")]
public class MaskedEdit_Date : AjaxControlToolkit.Testing.Client.MaskedEdit.MaskedEdit_Common
{
    public override IEnumerable<MaskedEditScenario> MaskedEdits
    {
        get 
        { 
            DatePage page = new DatePage();
            yield return new MaskedEditScenario(page.DateTextBoxMaskedEditExtender, "__/__/____", page.DateValidator);
            yield return new MaskedEditScenario(page.AutoCompleteDateMaskedEditExtender, "__/__/____", page.AutoCompleteMaskedEditValidator);
        }
    }

    /// <summary>
    /// Make sure that the Date textbox mask does not show when it does
    /// not have focus and shows when focus is set.
    /// </summary>
    [WebTestMethod]
    public void DateEmptyMask()
    {
        DatePage page = new DatePage();
        /* Date field tests */
        // Step 1 Valid date
        // Check mask before focus - it should be empty
        Assert.IsNull(page.DateTextBox.GetAttributes().Value);
        // Focus on date textbox to see mask
        page.DateTextBox.Focus();
        // validate mask
        Assert.AreEqual("__/__/____", page.DateTextBox.GetAttributes().Value ?? string.Empty);
    }

    /// <summary>
    /// Make sure that the empty field message shows when no date is entered.
    /// </summary>
    [WebTestMethod]
    public void DateEmptyValidator()
    {
        DatePage page = new DatePage();
        page.DateTextBox.Blur();
        // check message
        Assert.AreEqual("Please enter a date", page.DateValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Make sure valid dates are accepted and no validator messages popup.
    /// </summary>
    [WebTestMethod]
    public void ValidDate()
    {
        DatePage page = new DatePage();
        // Set full date 
        page.DateTextBox.SetText("12/31/2007", true);
        Assert.AreEqual("12/31/2007", page.DateTextBox.GetAttributes().Value ?? string.Empty);
        // check message
        Assert.AreEqual("", page.DateValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Make sure entering insufficient characters completes the rest of the date
    /// filling in the incomplete fields with that of the present date and that the
    /// final date is valid.
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.All, "The values are never set and just the mask is present.  It looks like MaskedEdit prevents them from being set.")]
    public void ValidPartialDate()
    {
        DatePage page = new DatePage();
        // Set partial date - this does not work
        page.DateTextBox.SetText("0512", true);
        Assert.AreEqual("05/12/" + DateTime.Today.Year, page.DateTextBox.GetAttributes().Value ?? string.Empty);
        // check message
        Assert.AreEqual("", page.DateValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Make sure entering insufficient characters and an empty mask 
    /// completes the rest of the date filling in the incomplete fields with 
    /// that of the present date and that the final date is valid.
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.All, "The values get set but the empty values are set to zeroes instead of the values associated with the \"Today\": 05/12/0000")]
    public void ValidPartialDateFullMask()
    {
        DatePage page = new DatePage();
        // Set partial date - this does not work
        page.DateTextBox.SetText("05/12/____", true);
        Assert.AreEqual("05/12/" + DateTime.Today.Year, page.DateTextBox.GetAttributes().Value ?? string.Empty);
        // check message
        Assert.AreEqual("", page.DateValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Bad date. Month out of range.
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.InternetExplorer7, "TODO: ???")]
    public void InvalidDate()
    {
        DatePage page = new DatePage();
        // Step 2 Invalid date
        page.DateTextBox.SetText("31/31/2007", true);
        // check date
        Assert.AreEqual("31/31/2007", page.DateTextBox.GetAttributes().Value ?? string.Empty);

        // check message
        Assert.AreEqual("Invalid date", page.DateValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Date beyond Maximum date allowed.
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.InternetExplorer7, "TODO: ???")]
    public void InvalidDateTooLate()
    {
        DatePage page = new DatePage();
        // Step 2 Invalid date
        page.DateTextBox.SetText("12/13/2999", true);
        // check date
        Assert.AreEqual("12/13/2999", page.DateTextBox.GetAttributes().Value ?? string.Empty);
        // check message
        Assert.AreEqual("Date too far out", page.DateValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Date earlier that Minimum date allowed.
    /// </summary>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.InternetExplorer7, "TODO: ???")]
    public void InvalidDateTooEarly()
    {
        DatePage page = new DatePage();
        // Step 2 Invalid date
        page.DateTextBox.SetText("01/01/1899", true);
        // check date
        Assert.AreEqual("01/01/1899", page.DateTextBox.GetAttributes().Value ?? string.Empty);
        // check message
        Assert.AreEqual("Date too early", page.DateValidator.GetAttributes().InnerText);
    }

    /// <summary>
    /// Validate that the specified autocomplete date is used
    /// to complete the date if the entire date value is not provided.
    /// </summary>
    /// <remarks>
    /// FAIL: actual: "05/00/1964"; Works fine when trying manually.
    /// second test actual: "12/12/0000"; Works fine manually.
    /// </remarks>
    [WebTestMethod]
    [WebTestFailureTag(Browsers.All, "Both 05/00/1964 and 12/12/0000 work fine manually")]
    public void AutoCompleteDate()
    {
        DatePage page = new DatePage();
        page.AutoCompleteDateTextBox.SetText("05/__/____", true);
        Assert.AreEqual("05/23/1964", page.AutoCompleteDateTextBox.GetAttributes().Value);
        page.AutoCompleteDateTextBox.SetText("12/12/____", true);
        Assert.AreEqual("12/12/1964", page.AutoCompleteDateTextBox.GetAttributes().Value);
    }

    /// <summary>
    /// Validate that invalid characters are filtered out.
    /// </summary>
    [WebTestMethod]
    public void InvalidChars()
    {
        DatePage page = new DatePage();
        // Step 2 Invalid date
        page.AutoCompleteDateTextBox.SetText("01a/02b/1980", true);
        // check date
        Assert.AreEqual("01/02/1980", page.AutoCompleteDateTextBox.GetAttributes().Value ?? string.Empty);
        // check message
        Assert.AreEqual(string.Empty, page.AutoCompleteMaskedEditValidator.GetAttributes().InnerText);
    }
}
