// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Web.UI.WebControls;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.DropDownExtender Model
    /// </summary>
    /// <TestComponent Name="DropDown">
    ///   <ToolkitType>AjaxControlToolkit.DropDownExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.DropDown.DropDown_Common</CommonTestSuite>
    /// </TestComponent>
    public class DropDownBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// DropDownControl
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> DropDownControl
        {
            get { return _dropDownControl; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _dropDownControl;

        /// <summary>
        /// HighlightBorderColor
        /// </summary>
        public BehaviorProperty<Color> HighlightBorderColor
        {
            get { return _highlightBorderColor; }
        }
        private BehaviorProperty<Color> _highlightBorderColor;

        /// <summary>
        /// HighlightBackColor
        /// </summary>
        public BehaviorProperty<Color> HighlightBackColor
        {
            get { return _highlightBackColor; }
        }
        private BehaviorProperty<Color> _highlightBackColor;

        /// <summary>
        /// DropArrowBackColor
        /// </summary>
        public BehaviorProperty<Color> DropArrowBackColor
        {
            get { return _dropArrowBackColor; }
        }
        private BehaviorProperty<Color> _dropArrowBackColor;

        /// <summary>
        /// DropArrowImageUrl
        /// </summary>
        public BehaviorProperty<string> DropArrowImageUrl
        {
            get { return _dropArrowImageUrl; }
        }
        private BehaviorProperty<string> _dropArrowImageUrl;

        /// <summary>
        /// DropArrowWidth
        /// </summary>
        public BehaviorProperty<Unit> DropArrowWidth
        {
            get { return _dropArrowWidth; }
        }
        private BehaviorProperty<Unit> _dropArrowWidth;

        /// <summary>
        /// DynamicControl
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> DynamicControl
        {
            get { return _dynamicControl; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _dynamicControl;

        /// <summary>
        /// DynamicContextKey
        /// </summary>
        public BehaviorProperty<string> DynamicContextKey
        {
            get { return _dynamicContextKey; }
        }
        private BehaviorProperty<string> _dynamicContextKey;

        /// <summary>
        /// DynamicServicePath
        /// </summary>
        public BehaviorProperty<string> DynamicServicePath
        {
            get { return _dynamicServicePath; }
        }
        private BehaviorProperty<string> _dynamicServicePath;

        /// <summary>
        /// DynamicServiceMethod
        /// </summary>
        public BehaviorProperty<string> DynamicServiceMethod
        {
            get { return _dynamicServiceMethod; }
        }
        private BehaviorProperty<string> _dynamicServiceMethod;

        /// <summary>
        /// CacheDynamicResults
        /// </summary>
        public BehaviorProperty<bool> CacheDynamicResults
        {
            get { return _cacheDynamicResults; }
        }
        private BehaviorProperty<bool> _cacheDynamicResults;

        /// <summary>
        /// IsOver
        /// </summary>
        public BehaviorProperty<bool> IsOver
        {
            get { return _isOver; }
        }
        private BehaviorProperty<bool> _isOver;

        /// <summary>
        /// IsOpen
        /// </summary>
        public BehaviorProperty<bool> IsOpen
        {
            get { return _isOpen; }
        }
        private BehaviorProperty<bool> _isOpen;

        /// <summary>
        /// DropWrapper
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> DropWrapper
        {
            get { return _dropWrapper; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _dropWrapper;

        /// <summary>
        /// SupportsContextMenu
        /// </summary>
        public BehaviorProperty<bool> SupportsContextMenu
        {
            get { return _supportsContextMenu; }
        }
        private BehaviorProperty<bool> _supportsContextMenu;
        
        /// <summary>
        /// HoverBehavior
        /// </summary>
        public HoverBehavior HoverBehavior
        {
            get { return _hoverBehavior; }
        }
        private HoverBehavior _hoverBehavior;

        /// <summary>
        /// PopupBehavior
        /// </summary>
        public PopupBehavior PopupBehavior
        {
            get 
            {
                if (_popupBehavior == null)
                {
                    _popupBehavior = new PopupBehavior(
                        _dropDownControl.Reference,
                        BehaviorID + "_dropPopupPopupBehavior",
                        string.Format("{0}._dropPopupPopupBehavior", BehaviorReferenceExpression),
                        Page);
                }
                return _popupBehavior; 
            }
        }
        private PopupBehavior _popupBehavior;

        /// <summary>
        /// AjaxControlToolkit.DropDownExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public DropDownBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _dropDownControl = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}._dropDownControl");
            _highlightBorderColor = BehaviorProperty<Color>.CreateProperty(this, "highlightBorderColor", ConvertToColor);
            _highlightBackColor = BehaviorProperty<Color>.CreateProperty(this, "highlightBackgroundColor", ConvertToColor);
            _dropArrowBackColor = BehaviorProperty<Color>.CreateProperty(this, "dropArrowBackgroundColor", ConvertToColor);
            _dropArrowImageUrl = BehaviorProperty<string>.CreateProperty(this, "dropArrowImageUrl");
            _dropArrowWidth = BehaviorProperty<Unit>.CreateProperty(this, "dropArrowWidth", ConvertToUnit);
            _dynamicControl = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "dynamicControlID");
            _dynamicContextKey = BehaviorProperty<string>.CreateProperty(this, "dynamicContextKey");
            _dynamicServicePath = BehaviorProperty<string>.CreateProperty(this, "dynamicServicePath");
            _dynamicServiceMethod = BehaviorProperty<string>.CreateProperty(this, "dynamicServiceMethod");
            _cacheDynamicResults = BehaviorProperty<bool>.CreateProperty(this, "cacheDynamicResults");
            _isOver = BehaviorProperty<bool>.CreateProperty(this, "isOver");
            _isOpen = BehaviorProperty<bool>.CreateProperty(this, "isOpen");
            _dropWrapper = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", ReadStrategy.Demand, WriteStrategy.Immediate, "{0}._dropWrapper");
            _supportsContextMenu = BehaviorProperty<bool>.CreateCustomProperty(this, null, "Sys.Browser.agent === Sys.Browser.InternetExplorer", "throw 'SupportsContextMenu is readonly';");
            _hoverBehavior = new HoverBehavior(
                                element.ParentElement,
                                behaviorID + "_dropWrapperHoverBehavior",
                                string.Format("{0}._dropWrapperHoverBehavior", BehaviorReferenceExpression),
                                page);
        }

        public void AssertClosed()
        {
            InvalidateAndRequire(IsOver, IsOpen);
            Assert.AreEqual(Display.None, DropDownControl.Reference.GetAttributes().Style.Display);
            Assert.IsFalse(IsOver.Value);
            Assert.IsFalse(IsOpen.Value);
        }

        public void AssertHover()
        {
            InvalidateAndRequire(IsOver, IsOpen);
            Assert.AreEqual(Display.None, DropDownControl.Reference.GetAttributes().Style.Display);
            Assert.IsTrue(IsOver.Value);
            Assert.IsFalse(IsOpen.Value);
        }

        public void AssertOpen(bool over)
        {
            InvalidateAndRequire(IsOver, IsOpen);
            Assert.AreNotEqual(Display.None, DropDownControl.Reference.GetAttributes().Style.Display);
            if (over)
            {
                Assert.IsTrue(IsOver.Value);
            }
            Assert.IsTrue(IsOpen.Value);
        }

        private Unit ConvertToUnit(object obj)
        {
            return ((null == obj) ? Unit.Empty : Unit.Parse(obj.ToString()));
        }

        private Color ConvertToColor(object obj)
        {
            WebColorConverter webColorConverter = new WebColorConverter();
            return ((null == obj) ? Color.Empty : (Color)webColorConverter.ConvertFromString(obj.ToString()));
        }
    }
}
